package ru.yandex.chemodan.app.webdav.repository.upload;

import java.io.InputStream;
import java.util.concurrent.atomic.AtomicReference;

import lombok.AllArgsConstructor;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.InputStreamEntity;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.webdav.servlet.PutResult;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.misc.bytes.ByteSequence;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author tolmalev
 */
@AllArgsConstructor
public class UploaderClient {
    private static final Logger logger = LoggerFactory.getLogger(UploaderClient.class);

    private final HttpClient httpClient;

    /**
     * @return status code
     */
    public PutResult sendToUploader(InputStream dataStream, Upload upload, String uploadUrl) {
        HttpPut httpPut = new HttpPut(uploadUrl);

        YandexCloudRequestIdHolder.getO().forEach(v -> httpPut.addHeader("Yandex-Cloud-Request-ID", v));
        upload.replaceMd5.forEach(v -> httpPut.addHeader("If-Match", v));
        upload.yandexDiff.forEach(v -> httpPut.addHeader("Yandex-Diff", v));
        upload.contentRange.forEach(v -> httpPut.addHeader("Content-Range", v.toString()));
        // can't use entity.setContentType because of OOM in RequestBuilder.doCopy (because of applyThreadLocalTimeout)
        upload.contentType.forEach(v -> httpPut.addHeader("Content-Type", v));

        httpPut.addHeader("X-Forwarded-For", upload.remoteIpAdress.format());

        Instant uploadStart = Instant.now();
        AtomicReference<Instant> uploadComplete = new AtomicReference<>();

        InputStreamEntity entity = new InputStreamEntity(IoUtils.intercept(dataStream, new Function1V<ByteSequence>() {
            private long streamed = 0;
            private long chunk = DataSize.fromMegaBytes(100).toBytes();

            @Override
            public void apply(ByteSequence byteSequence) {
                if (streamed == 0) {
                    String sizeStr =
                            upload.contentLength.map(DataSize::fromBytes).map(DataSize::toPrettyString).getOrElse("unknown");
                    logger.debug("Start streaming {} to {}. mode={}. "
                                    + "If-Match: {}, Yandex-Diff: {}, Content-Length: {}, Content-Range: {}, Content-Type: {}",
                            sizeStr, uploadUrl, upload.mode,
                            upload.replaceMd5, upload.yandexDiff, upload.size.getOrElse(-1L), upload.contentRange, upload.contentType);
                }
                long old = streamed;
                streamed += byteSequence.length();

                if (streamed / chunk > old / chunk) {
                    Duration duration = new Duration(uploadStart, Instant.now());
                    String speedStr;
                    if (duration.getStandardSeconds() == 0) {
                        speedStr = "infinity";
                    } else {
                        speedStr = Math.round(streamed / 1024.0 / duration.getStandardSeconds()) + " KB/s";
                    }

                    logger.debug("Streamed {} ({}) as speed={}",
                            DataSize.fromBytes(streamed).toPrettyString(),
                            streamed,
                            speedStr);
                }
                if (upload.contentLength.isSome(streamed)) {
                    logger.info("Fully Streamed {} ({}). time={}. Wait for uploader response now.",
                            DataSize.fromBytes(streamed).toPrettyString(), streamed,
                            TimeUtils.secondsStringToNow(uploadStart)
                    );
                    uploadComplete.set(Instant.now());
                }
            }
        }), upload.contentLength.getOrElse(-1L));

        upload.contentEncoding.forEach(entity::setContentEncoding);

        httpPut.setEntity(entity);

        WhatThreadDoes.Handle wtdState = WhatThreadDoes.push(""
                + "Uploading file to " + uploadUrl
                + ". Start time = " + Instant.now()
        );

        try {
            return ApacheHttpClientUtils.execute(httpPut, httpClient, response -> {
                int statusCode = response.getStatusLine().getStatusCode();
                Option<String> location = Cf.x(response.getHeaders("Location")).singleO().map(Header::getValue);
                Instant instant = uploadComplete.get();

                String waitTimeStr = "unknown";
                if (instant != null) {
                    waitTimeStr = TimeUtils.secondsStringToNow(instant);
                }
                String totalTimeStr = TimeUtils.secondsStringToNow(uploadStart);

                logger.info("Received status={} from uploder. total={}, response_wait={}",
                        statusCode, totalTimeStr, waitTimeStr
                );
                return new PutResult(statusCode, location);
            });
        } catch (HttpException e) {
            wtdState.popSafely();
            if (!e.getStatusCode().isPresent() && e.getCause() instanceof ClientProtocolException) {
                throw new HttpException(499, "Client has disconnected", e.getCause());
            }
            throw e;
        }
    }

    public long getUploadedSize(String uploadUrl) {
        return ApacheHttpClientUtils.execute(new HttpHead(uploadUrl), httpClient,
                new Abstract200ExtendedResponseHandler<Long>() {
                    @Override
                    protected Long handle200Response(HttpResponse response) {
                        Header header = response.getFirstHeader("Content-Length");
                        if (header == null) {
                            throw new PermanentHttpFailureException("Not found", HttpStatus.SC_404_NOT_FOUND);
                        }
                        return Cf.Long.parseSafe(header.getValue()).getOrThrow(
                                () -> new PermanentHttpFailureException("Not found", HttpStatus.SC_404_NOT_FOUND)
                        );
                    }
                });
    }
}
