package ru.yandex.chemodan.app.webdav.servlet;

import java.util.concurrent.atomic.AtomicReference;

import javax.servlet.AsyncContext;

import lombok.AllArgsConstructor;
import lombok.Getter;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.filter.AuthenticationFilter;
import ru.yandex.chemodan.app.webdav.repository.ComplexCallback;
import ru.yandex.chemodan.app.webdav.repository.MpfsCallbackBase;
import ru.yandex.chemodan.app.webdav.repository.MpfsResource;
import ru.yandex.chemodan.app.webdav.repository.MpfsResourceManager;
import ru.yandex.chemodan.app.webdav.repository.PathUtils;
import ru.yandex.chemodan.mpfs.MpfsOperation;

/**
 * @author tolmalev
 */
public abstract class DavHandlerBase implements DavMethodHandler {
    private static final String CREATE_DIRS_HEADER = "X-Yandex-Create";
    private static final Duration ASYNC_CONTEXT_TIMEOUT = Duration.standardMinutes(30);

    protected final MpfsResourceManager manager;
    protected final String method;

    protected DavHandlerBase(MpfsResourceManager manager, String method) {
        this.manager = manager;
        this.method = method;
    }

    @Override
    public String method() {
        return method;
    }

    private AsyncContext startAsync(WebdavRequest request) {
        AsyncContext ctx = ((WebDavRequestSupport) request).getHttpRequest().startAsync();
        ctx.setTimeout(ASYNC_CONTEXT_TIMEOUT.getMillis());
        return ctx;
    }

    protected AuthInfo getAuthInfo(WebdavRequest request) {
        return AuthenticationFilter.getAuthInfo(request);
    }

    SourcePaths extractSourcePaths(WebdavRequest request, MpfsResource resource) throws DavException {
        Option<MultipleRequest> multipleRequest = MultipleRequest.parse(request);

        ListF<String> sourcePaths = multipleRequest
                .flatMap(m -> PathUtils.getSourcePaths(m, resource.getRealPath()));

        return sourcePaths.isEmpty() ? SourcePaths.single(resource.getRealPath()) : SourcePaths.multiple(sourcePaths);
    }

    Tuple2List<String, String> zipWithDestPath(
            ListF<String> sourcePaths, MpfsResource destResource, boolean isMultiple)
    {
        if (!isMultiple) {
            return sourcePaths.zip(Cf.list(destResource.getRealPath()));
        } else {
            return sourcePaths.zipWith(src ->
                    PathUtils.childPath(destResource.getRealPath(), PathUtils.getDisplayName(src))
            );
        }
    }

    void withMpfsAsyncOperation(
            WebdavRequest request,
            Function2<AsyncContext, AtomicReference<String>, MpfsOperation> op)
    {
        AsyncContext ctx = startAsync(request);

        AtomicReference<String> operationId = new AtomicReference<>();
        MpfsOperation operation = op.apply(ctx, operationId);
        operationId.set(operation.getOid());
    }

    void applySingleOrComplex(
            WebdavRequest request, ListF<ComplexCallback.AsyncOperation> operations, boolean isMultiple)
    {
        if (isMultiple) {
            AsyncContext ctx = startAsync(request);
            new ComplexCallback(manager, ctx, operations).finishOrCallNextOperation();
        } else {
            withMpfsAsyncOperation(request, (ctx, operationId) -> {
                        ComplexCallback.AsyncOperation single = operations.single();
                        return single.fn.apply(ctx, operationId, new MpfsCallbackBase(ctx, operationId, single.okCode));
                    }
            );
        }
    }

    boolean doMkdirs(WebdavRequest request) {
        return Option.ofNullable(request.getHeader(CREATE_DIRS_HEADER)).isSome("T");
    }

    @AllArgsConstructor
    @Getter
    static class SourcePaths {
        private final ListF<String> paths;
        private final boolean isMultiple;

        static SourcePaths single(String path) {
            return new SourcePaths(Cf.list(path), false);
        }

        static SourcePaths multiple(ListF<String> paths) {
            return new SourcePaths(paths, true);
        }

    }
}
