package ru.yandex.chemodan.app.webdav.servlet;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.Optional;

import org.apache.http.client.HttpClient;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.webdav.repository.MpfsResourceManager;
import ru.yandex.chemodan.app.webdav.repository.upload.DavUploaderContextConfiguration;
import ru.yandex.chemodan.app.webdav.repository.upload.UploaderClient;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsClientImpl;
import ru.yandex.chemodan.mpfs.MpfsUserInitParamsExtractor;
import ru.yandex.chemodan.mulca.MulcaClientContextConfiguration;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.chemodan.util.ping.CoolPingServlet;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.regex.Pattern2;

/**
 * @author tolmalev
 */
@Import(DavUploaderContextConfiguration.class)
@Configuration
public class DavHandlersContextConfiguration {
    @Bean
    public ProppatchHandler proppatchHandler(MpfsResourceManager manager) {
        return new ProppatchHandler(manager);
    }

    @Bean
    public RedirectToHelpHandler redirectToHelpHandler() {
        return new RedirectToHelpHandler();
    }

    @Bean
    public UserinfoHandler userinfoHandler(
            MpfsClient mpfsClient,
            @Value("${webdav.client.upload.concurrency}")
            int uploadСoncurrency,
            @Value("${webdav.client.deny.rules}")
            String denyRulesStr)
    {
        // XXX: very dab hack :(
        return new UserinfoHandler(mpfsClient, uploadСoncurrency,
                Cf.list(denyRulesStr.split("\\},\\{")).map(str -> {
                    if (!str.startsWith("{")) {
                        str = "{" + str;
                    }
                    if (!str.endsWith("}")) {
                        str += "}";
                    }

                    return UserinfoHandler.DenyRule.P.parseJson(str);
                })
        );
    }

    @Bean
    public PingHandler pingHandler(CoolPingServlet coolPingServlet) {
        return new PingHandler(coolPingServlet);
    }

    @Bean
    public InstallHandler installHandler(MpfsClient mpfsClient) {
        return new InstallHandler(mpfsClient);
    }

    @Bean
    public OperationsHandler operationsHandler(MpfsClient mpfsClient) {
        return new OperationsHandler(mpfsClient);
    }

    @Bean
    @MpfsIndex
    @OverridableValuePrefix("mpfs.index")
    public HttpClientConfigurator mpfsIndexHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Qualifier
    @Retention(RetentionPolicy.RUNTIME)
    public @interface MpfsIndex {
    }

    @Bean
    public IndexHandler indexHandler(
            @Value("${mpfs.host}") String host,
            @Value("${mpfs.index.init-on-demand}") boolean initOnDemand,
            @MpfsIndex HttpClientConfigurator configurator,
            Optional<MpfsUserInitParamsExtractor> userInitParamsExtractor,
            @Value("${djfs.host:-}") String djfsHost,
            @Qualifier("djfs") Optional<HttpClientConfigurator> djfsHttpClientConfigurator,
            @Value("${mpfs.ignore-user-not-initialized-exception}") boolean ignoreUserNotInitializedException)
    {
        HttpClient httpClient = configurator.configure();

        return new IndexHandler(new MpfsClientImpl(httpClient, httpClient, host,
                userInitParamsExtractor.orElse(MpfsUserInitParamsExtractor.empty()), initOnDemand,
                StringUtils.isBlank(djfsHost) ? Option.empty() : Option.of(djfsHost),
                Option.x(djfsHttpClientConfigurator.map(HttpClientConfigurator::configure)),
                ignoreUserNotInitializedException));
    }

    @Bean
    public GetHandler getHandler(
            MulcaClient mulcaClient,
            @Value("${get.redirect.threshold}")
            DataSize redirectGetThreshold,
            @MulcaClientContextConfiguration.Mulca
            HttpClient httpClient)
    {
        return new GetHandler(redirectGetThreshold, mulcaClient, httpClient);
    }

    @Bean
    public GetDigestHandler getDigestHandler(
                    MulcaClient mulcaClient,
                    @Value("${get.redirect.threshold}")
                    DataSize redirectGetThreshold,
                    @MulcaClientContextConfiguration.Mulca
                    HttpClient httpClient)
    {
        return new GetDigestHandler(redirectGetThreshold, mulcaClient, httpClient);
    }

    @Bean
    public TrashCleanHandler trashCleanHandler(MpfsResourceManager manager) {
        return new TrashCleanHandler(manager);
    }

    @Bean
    public GetPreviewHandler getPreviewHandler(MpfsClient mpfsClient) {
        return new GetPreviewHandler(mpfsClient, previewHttpClientConfigurator().configure());
    }

    @Bean
    @OverridableValuePrefix("preview")
    public HttpClientConfigurator previewHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    public RejectInviteHandler rejectInviteHandler(MpfsClient mpfsClient) {
        return new RejectInviteHandler(mpfsClient);
    }

    @Bean
    public AcceptInviteHandler acceptInviteHandler(MpfsClient mpfsClient) {
        //todo property
        return new AcceptInviteHandler(mpfsClient, "https://webdav.yandex.ru/");
    }

    @Bean
    public HeadHandler headHandler() {
        return new HeadHandler();
    }

    @Bean
    public HeadForUploadHandler headForUploadHandler(MpfsClient mpfsClient, UploaderClient uploaderClient) {
        return new HeadForUploadHandler(mpfsClient, uploaderClient);
    }

    @Bean
    public PropfindHandler propfindHandler(MpfsResourceManager manager) {
        return new PropfindHandler(manager);
    }

    @Bean
    public OptionsHandler optionsHandler() {
        return new OptionsHandler();
    }

    @Bean
    public DeleteHandler deleteHandler(MpfsResourceManager manager) {
        return new DeleteHandler(manager);
    }

    @Bean
    public CopyHandler copyHandler(MpfsResourceManager manager) {
        return new CopyHandler(manager);
    }

    @Bean
    public PublishHandler publishHandler(MpfsResourceManager manager) {
        return new PublishHandler(manager);
    }

    @Bean
    public PushHandler pushHandler(MpfsResourceManager manager) {
        return new PushHandler(manager);
    }

    @Bean
    public MoveHandler moveHandler(MpfsResourceManager manager) {
        return new MoveHandler(manager);
    }

    @Bean
    public MkColHandler mkColHandler(MpfsResourceManager manager) {
        return new MkColHandler(manager);
    }

    @Bean
    public PutHandler putHandler(
            UploaderClient uploaderClient,
            MpfsResourceManager manager,
            @Value("${put.redirect.threshold}")
            DataSize redirectPutThreshold,
            @Value("${webdav.autohide.pattern}")
            String invisiblePatternStr)
    {
        return new PutHandler(manager, redirectPutThreshold, uploaderClient, Pattern2.compile(invisiblePatternStr));
    }

    @Bean
    public LockHandler lockHandler() {
        return new LockHandler();
    }

    @Bean
    public UnlockHandler unlockHandler() {
        return new UnlockHandler();
    }
}
