package ru.yandex.chemodan.app.webdav.servlet;

import java.io.IOException;
import java.io.InputStream;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.AllArgsConstructor;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.util.EntityUtils;
import org.apache.jackrabbit.webdav.DavConstants;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.filter.AuthenticationFilter;
import ru.yandex.chemodan.app.webdav.repository.MpfsResource;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaException;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
@AllArgsConstructor
public class GetHandler implements DavMethodHandler {
    private static final Logger logger = LoggerFactory.getLogger(GetHandler.class);

    private final DataSize redirectGetThreshold;
    private final MulcaClient mulcaClient;
    private final HttpClient httpClient;

    @Override
    public void handle(WebdavRequest request, WebdavResponse response, MpfsResource resource)
            throws IOException, DavException
    {
        if (resource.isCollection()) {
            throw new DavException(HttpStatus.SC_415_UNSUPPORTED_MEDIA_TYPE);
        }
        if (!resource.exists()) {
            throw new DavException(HttpStatus.SC_404_NOT_FOUND);
        }

        MpfsFileInfo info = resource.getFileInfo();
        if (!resource.isCollection()) {
            response.addHeader("Accept-Ranges", "bytes");
            info.getMeta().getMimeType().forEach(response::setContentType);
            info.getMeta().getMd5().forEach(v -> response.addHeader("ETag", v));
        }
        response.addDateHeader(DavConstants.HEADER_LAST_MODIFIED, info.times.mtime * 1000);

        if (info.getMeta().getSize().map(DataSize::toBytes).getOrElse(0L) == 0) {
            // do nothing for zero-length file
            response.addHeader("Content-Length", "0");
            return;
        }

        HttpServletRequestX reqX = HttpServletRequestX.wrap(request);

        if (reqX.getHeaderO("If-None-Match").isSome(info.getMeta().getMd5().getOrElse("unknown"))) {
            response.addHeader("Content-Length", "0");
            response.setStatus(HttpStatus.SC_304_NOT_MODIFIED);
            return;
        }

        Option<String> rangeHeader = reqX.getHeaderO("Range").filter(range -> !"bytes=0-".equals(range.trim()));

        if (isFullFile(rangeHeader) && allowGetRedirect(info, AuthenticationFilter.getAuthInfo(request))) {
            String redirectUrl = info.getMeta().getFileUrl().get();
            response.addHeader("Content-Length", "0");
            response.addHeader("Location", redirectUrl);
            response.setStatus(HttpStatus.SC_302_MOVED_TEMPORARILY);
            logger.debug("Redirect for GET: {}", redirectUrl);
            return;
        }

        WhatThreadDoes.Handle handle = WhatThreadDoes.push(""
                + "Downloading file " + info.getMeta().getFileMid().get()
                + ". Range = " + rangeHeader
                + ". Start time = " + Instant.now());
        try {
            spoolFileContent(info,
                    MulcaId.fromSerializedString(info.getMeta().getFileMid().get()), request, response, rangeHeader);
        } finally {
            handle.popSafely();
        }

        response.flushBuffer();
    }

    private boolean isFullFile(Option<String> rangeHeader) {
        return !rangeHeader.isPresent();
    }

    @Override
    public String method() {
        return "GET";
    }

    void spoolFileContent(MpfsFileInfo info,
            MulcaId mulcaId,
            HttpServletRequest req,
            HttpServletResponse resp,
            Option<String> rangeHeader)
    {
        HttpGet request = new HttpGet(mulcaClient.getDownloadUri(mulcaId));

        Option<String> isRangeO = HttpServletRequestX.wrap(req).getHeaderO("If-Range");

        if (!rangeHeader.isPresent()
                // md5 != if-range
                || (isRangeO.isPresent() && !isRangeO.isSome(info.getMeta().getMd5().get())))
        {
            // no nothing
            // not range request
            // so don't add header Range to mulcagate request
        } else {
            request.addHeader("Range", rangeHeader.get());
        }

        // XXX: copy-paste of ApacheHttpClient4InputStreamSource

        InputStream inputStream = null;
        HttpEntity entity = null;

        try {
            HttpResponse httpResponse = httpClient.execute(request);
            int code = httpResponse.getStatusLine().getStatusCode();
            entity = httpResponse.getEntity();
            if (code != HttpStatus.SC_200_OK && code != HttpStatus.SC_206_PARTIAL_CONTENT) {
                throw ExceptionUtils.throwException(new DavException(code));
            }

            resp.setStatus(code);

            if (httpResponse.getFirstHeader("Content-Range") != null) {
                resp.addHeader("Content-Range", httpResponse.getFirstHeader("Content-Range").getValue());
            }
            if (httpResponse.getFirstHeader("Content-Length") != null) {
                resp.addHeader("Content-Length", httpResponse.getFirstHeader("Content-Length").getValue());
            }
            if (httpResponse.getFirstHeader("Content-Type") != null) {
                String contentType = httpResponse.getFirstHeader("Content-Type").getValue();
                if (contentType.startsWith("multipart/byteranges")) {
                    resp.setContentType(contentType);
                }
            }

            inputStream = entity.getContent();
            IoUtils.copy(inputStream, resp.getOutputStream());
        } catch (IOException e) {
            throw new MulcaException(HttpStatus.SC_500_INTERNAL_SERVER_ERROR, "failed to execute " + this + ": " + e, e);
        } finally {
            // if inputStream is not null, releaseConnection happens inside
            // EntityInputStream
            try {
                request.releaseConnection();
                logger.trace("Connection released");
            } catch (Exception e) {
                logger.error("Ignored Exception: {}", e);
            }

            if (inputStream != null) {
                try {
                    inputStream.close();
                    logger.trace("inputStream closed");
                } catch (Exception e) {
                    logger.error("Ignored Exception: {}", e);
                }
            }
            if (inputStream == null) {
                try {
                    EntityUtils.consume(entity);
                    logger.trace("entity consumed");
                } catch (IOException e) {
                    logger.error("Ignored Exception: {}", e);
                }
            }
        }
    }

    private boolean allowGetRedirect(MpfsFileInfo info, AuthInfo authInfo) {
        return authInfo.clientCapabilities.isGetRedirect()
                && info.getMeta().getSize().getOrElse(DataSize.ZERO).gt(redirectGetThreshold);
    }

}
