package ru.yandex.chemodan.app.webdav.servlet;

import java.io.IOException;
import java.util.regex.Matcher;

import javax.annotation.Nullable;

import lombok.AllArgsConstructor;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.auth.OurClient;
import ru.yandex.chemodan.app.webdav.repository.MpfsResource;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.misc.io.OutputStreamOutputStreamSource;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.ReadToOutputStreamSourceResponseHandler;
import ru.yandex.misc.regex.Pattern2;

/**
 * @author tolmalev
 */
@AllArgsConstructor
public class GetPreviewHandler implements DavMethodHandler {
    private final MpfsClient mpfsClient;
    private final HttpClient previewHttpClient;

    private static final MapF<String, Integer> sizes = Tuple2List.<String, Integer>fromPairs(
            "XXXL", 1280,
            "XXL", 1024,
            "XL", 800,
            "L", 500,
            "M", 300,
            "S", 150,
            "XS", 100,
            "XXS", 75,
            "XXXS", 50
    ).toMap().unmodifiable();

    private static final Pattern2 sizePattern = Pattern2.compile("(\\d*)x(\\d*)");

    @Override
    public void handle(WebdavRequest request, WebdavResponse response, MpfsResource resource)
            throws IOException, DavException
    {
        PreviewRequest previewRequest = parsePreviewRequest(request);

        MapF<String, Object> params = Cf.hashMap();
        params.put("preview_size", previewRequest.getPreviewSizeForMpfs());
        if (previewRequest.crop) {
            params.put("preview_crop", 1);
        }
        if (previewRequest.animate) {
            params.put("preview_animate", 1);
        }
        previewRequest.type.forEach(v -> params.put("preview_type", v));
        MpfsFileInfo info =
                mpfsClient.getFileInfoByUidAndPath(resource.getUser(), resource.getRealPath(), params, Cf.list("custom_preview", "mimetype"));

        //TODO: check etag
        if (!info.getMeta().getCustomPreview().isPresent()) {
            throw new DavException(HttpStatus.SC_404_NOT_FOUND);
        }

        String previewUrl = info.getMeta().getCustomPreview().get();

        info.getMeta().getMimeType().forEach(response::setContentType);

        if (getAuthInfo(request).clientCapabilities.isGetRedirect()) {
            response.sendRedirect(previewUrl);
        } else {
            HttpGet httpGet = new HttpGet(previewUrl);
            httpGet.setHeader("Authorization", request.getHeader("Authorization"));

            ApacheHttpClientUtils.execute(httpGet, previewHttpClient,
                    new ReadToOutputStreamSourceResponseHandler(new OutputStreamOutputStreamSource(response.getOutputStream()))
            );
        }
    }

    private PreviewRequest parsePreviewRequest(WebdavRequest request) throws DavException {
        if (request.getParameter("size") == null) {
            throw new DavException(HttpStatus.SC_400_BAD_REQUEST);
        }

        Tuple2<Integer, Integer> parsedSize = parseSize(request.getParameter("size"));
        Option<String> cropO = Option.ofNullable(request.getParameter("crop"));
        Option<String> animateO = Option.ofNullable(request.getParameter("animate"));
        Option<String> typeO = Option.ofNullable(request.getParameter("type"));

        boolean crop = !cropO.isSome("false");

        PreviewRequest result = new PreviewRequest();
        result.width = parsedSize._1;
        result.height = parsedSize._2;
        result.animate = animateO.isSome("true") || animateO.isSome("1");
        result.type = typeO;

        AuthInfo authInfo = getAuthInfo(request);
        if (authInfo.ourClient.filter(c -> c.type == OurClient.Type.DESKTOP).isPresent()
                && parsedSize._1 != null
                && parsedSize._2 != null
                && (parsedSize._1 <= 100 || parsedSize._2 <= 100))
        {
            result.crop = true;
        } else if (parsedSize._1 == null || parsedSize._2 == null) {
            result.crop = false;
        } else {
            result.crop = crop;
        }

        Option.ofNullable(request.getParameter("quality")).forEach(quality ->
                result.quality = Cf.Integer.parseSafe(quality));

        return result;
    }

    private Tuple2<Integer, Integer> parseSize(String size) throws DavException {
        try {
            if (sizes.containsKeyTs(size)) {
                return Tuple2.tuple(sizes.getTs(size), sizes.getTs(size));
            }

            Matcher matcher = sizePattern.matcher(size);
            if (matcher.matches()) {
                return Tuple2.tuple(parseSizePart(matcher.group(1)), parseSizePart(matcher.group(2)));
            }

            return Tuple2.tuple(Cf.Integer.parse(size), null);
        } catch (RuntimeException e) {
            throw new DavException(HttpStatus.SC_400_BAD_REQUEST);
        }
    }

    @Nullable
    private Integer parseSizePart(String str) {
        return Cf.Integer.parseSafe(str).getOrNull();
    }

    private static class PreviewRequest {
        public Integer width;
        public Integer height;
        public boolean crop;
        Option<Integer> quality;
        boolean animate;
        public Option<String> type;

        String getPreviewSizeForMpfs() {
            return getPreviewSizeForMpfs(width) + "x" + getPreviewSizeForMpfs(height);
        }

        private String getPreviewSizeForMpfs(Integer size) {
            return Option.ofNullable(size).map(Object::toString).getOrElse("");
        }
    }

    @Override
    public String method() {
        return "GET";
    }

    @Override
    public int order() {
        return 1;
    }

    @Override
    public boolean matches(WebdavRequest request, MpfsResource resource) {
        return Option.ofNullable(request.getQueryString()).filter(qs -> qs.startsWith("preview")).isPresent();
    }
}
