package ru.yandex.chemodan.app.webdav.servlet;

import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;

import javax.servlet.AsyncContext;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.SneakyThrows;
import org.apache.jackrabbit.webdav.DavConstants;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;
import org.apache.jackrabbit.webdav.WebdavResponseImpl;
import org.jetbrains.annotations.NotNull;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.mock.web.MockAsyncContext;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.auth.ClientCapabilities;
import ru.yandex.chemodan.app.webdav.auth.OurClient;
import ru.yandex.chemodan.app.webdav.callback.MpfsCallbacksContextConfiguration;
import ru.yandex.chemodan.app.webdav.callback.MpfsCallbacksManager;
import ru.yandex.chemodan.app.webdav.callback.MpfsOperationCallbackData;
import ru.yandex.chemodan.app.webdav.filter.AuthenticationFilter;
import ru.yandex.chemodan.app.webdav.repository.ComplexCallback;
import ru.yandex.chemodan.app.webdav.repository.MpfsCallbackBase;
import ru.yandex.chemodan.app.webdav.repository.MpfsResource;
import ru.yandex.chemodan.app.webdav.repository.MpfsResourceManager;
import ru.yandex.chemodan.app.webdav.repository.PathUtils;
import ru.yandex.chemodan.app.webdav.repository.properties.DavPropertiesContextConfiguration;
import ru.yandex.chemodan.app.webdav.repository.properties.PropertiesFactory;
import ru.yandex.chemodan.app.webdav.repository.properties.PropertiesSetter;
import ru.yandex.chemodan.boot.CoolPingServletContextConfiguration;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsClientContextConfiguration;
import ru.yandex.chemodan.mpfs.MpfsOperation;
import ru.yandex.chemodan.mulca.MulcaClientContextConfiguration;
import ru.yandex.chemodan.util.test.AbstractTest;
import ru.yandex.chemodan.util.test.HttpRecorderRule;
import ru.yandex.chemodan.util.test.MpfsMockConfiguration;
import ru.yandex.chemodan.util.test.MulcaMockConfiguration;
import ru.yandex.chemodan.util.test.TestUser;
import ru.yandex.chemodan.util.test.UploaderClientMockConfiguration;
import ru.yandex.chemodan.util.web.CustomYcridPrefixResolver;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.web.servlet.mock.MockHttpServletRequest;
import ru.yandex.misc.web.servlet.mock.MockHttpServletResponse;

@ContextConfiguration(classes = {
        DavHandlersContextConfiguration.class,
        MulcaClientContextConfiguration.class,
        UploaderClientMockConfiguration.class,
        MpfsClientContextConfiguration.class,
        MpfsIndexMockConfiguration.class,
        MpfsCallbacksContextConfiguration.class,
        DavPropertiesContextConfiguration.class,
        MpfsMockConfiguration.class,
        MulcaMockConfiguration.class,
        HandlersTest.Config.class
})
public class HandlersTest extends AbstractTest {

    private static final AuthInfo DEFAULT_AUTH = auth(ClientCapabilities.getDefault());
    private static final AuthInfo ROOT_AUTH = auth(new ClientCapabilities(false, false, false, false, "/", false));

    private static AuthInfo auth(ClientCapabilities capabilities) {
        return new AuthInfo(
                Option.of(TestUser.uid),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                false,
                Option.empty(),
                Option.of(OurClient.AUTOTESTS),
                capabilities,
                Option.empty(),
                Option.empty(),
                AuthInfo.AuthType.OAUTH,
                false
        );
    }

    @Configuration
    @Import({
            CoolPingServletContextConfiguration.class
    })
    public static class Config {

        @Bean
        public MpfsResourceManager diskDavResourceFactory(
                MpfsClient client,
                MpfsCallbacksManager callbacksManager,
                PropertiesSetter propertiesSetter,
                List<PropertiesFactory> propertiesFactories)
        {
            return new MpfsResourceManager(client, callbacksManager, propertiesSetter, Cf.x(propertiesFactories));
        }

        @Bean
        public PropertiesSetter propertiesSetter(MpfsClient mpfsClient) {
            return new PropertiesSetter(mpfsClient);
        }

        @Bean
        public CustomYcridPrefixResolver customYcridPrefixResolver() {
            return new WebdavYcridPrefixResolver();
        }

        @Bean
        public WebDavServlet wedDavServlet(MpfsResourceManager mpfsResourceManager, List<DavMethodHandler> handlers) {
            return new WebDavServlet(mpfsResourceManager, Cf.x(handlers)) {
                @NotNull
                @Override
                @SneakyThrows
                WebdavResponseImpl toWebdavResponse(HttpServletRequest request, HttpServletResponse response) {
                    WebdavResponseImpl webdavResponse = Mockito.spy(super.toWebdavResponse(request, response));
                    Mockito.doAnswer(invocation -> {
                        throw invocation.<DavException>getArgument(0);
                    }).when(webdavResponse).sendError(Mockito.any());
                    return webdavResponse;
                }
            };
        }

        @Bean
        public HttpRecorderRule httpRecorderRule() {
            return HttpRecorderRule.builder()
                    .prefix(System.getProperty("user.home") + "/arcadia/disk/webdav/src/test/resources/").build();
        }

    }

    @Rule
    @Autowired
    public final HttpRecorderRule httpRecorderRule = null;

    @Autowired
    private WebDavServlet servlet;

    @Autowired
    private MpfsClient mpfsClient;

    @Autowired
    private MpfsCallbacksManager callbacksManager;

    @Autowired
    private MpfsResourceManager resourceManager;

    @Autowired
    private CustomYcridPrefixResolver prefixResolver;

    @Autowired
    private List<DavMethodHandler> handlers;

    @Test
    public void testCallbackManager() {
        AtomicReference<String> id = new AtomicReference<>("42");
        AsyncContext ctx = new MockAsyncContext(null, new MockHttpServletResponse());
        String rid = callbacksManager.registerCallback(true, new MpfsCallbackBase(ctx, id, 200));
        MpfsOperationCallbackData data = new MpfsOperationCallbackData("200", Option.empty());
        callbacksManager
                .callbackReceived(rid, new String(MpfsOperationCallbackData.PS.getSerializer().serializeJson(data)));
    }

    @Test
    public void testComplexCallback() {
        AsyncContext ctx = new MockAsyncContext(null, new MockHttpServletResponse());
        ComplexCallback callback = new ComplexCallback(resourceManager, ctx, Cf.list(
                new ComplexCallback.AsyncOperation("http://ya.ru/", 200, (x, y, z) -> new MpfsOperation() {
                    @Override
                    public String getOid() {
                        return y.get();
                    }
                })
        ));
        callback.finishOrCallNextOperation();
        String rid = callbacksManager.registerCallback(true, callback);
        MpfsOperationCallbackData data = new MpfsOperationCallbackData("200", Option.empty());
        callbacksManager
                .callbackReceived(rid, new String(MpfsOperationCallbackData.PS.getSerializer().serializeJson(data)));
    }

    private static <T> void assertEnumerationEquals(Enumeration<T> a, Enumeration<T> b) {
        List<T> x = Collections.list(a);
        List<T> y = Collections.list(b);
        Assert.isTrue(x.containsAll(y) && y.containsAll(x));
    }

    @Test
    public void testPrefixResolver() {
        Assert.A.equals("dav", prefixResolver.getPrefix(new MockHttpServletRequest()));
        Assert.A.equals("sdk", prefixResolver.getPrefix(new MockHttpServletRequest() {{
            addHeader("X-Yandex-Sdk-Version", "42");
        }}));
    }

    @Test
    public void testWebdavRequestSupport() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        WebdavRequest support = servlet.toWebdavRequest(request);
        assertEnumerationEquals(request.getHeaderNames(), support.getHeaderNames());
        assertEnumerationEquals(request.getAttributeNames(), support.getAttributeNames());
        assertEnumerationEquals(request.getParameterNames(), support.getParameterNames());
        assertEnumerationEquals(request.getLocales(), support.getLocales());
    }

    @SneakyThrows
    private void checkHandler(HttpServletRequest request, HttpServletResponse response,
            Class<? extends DavMethodHandler> klass)
    {
        WebdavRequest webdavRequest = servlet.toWebdavRequest(request);
        WebdavResponse webdavResponse = servlet.toWebdavResponse(request, response);
        MpfsResource resource = servlet.getResource(webdavRequest, webdavResponse);
        Option<DavMethodHandler> handler =
                Option.ofNullable(klass).map(k -> Cf.x(handlers).filter(h -> h.getClass().equals(klass)).first());
        Assert.A.isTrue(handler.get().matches(webdavRequest, resource));
        Assert.A.equals(klass, servlet.findHandler(webdavRequest, resource).map(Object::getClass).getOrNull());
    }

    @SneakyThrows
    private void testHandler(
            AuthInfo auth,
            Class<? extends DavMethodHandler> klass,
            String resource,
            Consumer<MockHttpServletRequest> configurator,
            Consumer<MockHttpServletResponse> checker)
    {
        Option<DavMethodHandler> handler = Option.ofNullable(klass).map(k -> Cf.x(handlers)
                .filter(h -> h.getClass().equals(klass)).first());
        new MockHttpServletResponse() {{
            String href = StringUtils.removeStart(resource, auth.clientCapabilities.getBaseLocation());
            MockHttpServletRequest request =
                    new MockHttpServletRequest(handler.map(DavMethodHandler::method).getOrElse("GET"),
                            "/".equals(href) ? href : PathUtils.encodePath(href))
                    {
                        {
                            setAttribute(AuthenticationFilter.INFO_ATTR, auth);
                            configurator.accept(this);
                        }

                        @Override
                        public AsyncContext startAsync(ServletRequest servletRequest, ServletResponse servletResponse) {
                            return new MockAsyncContext(servletRequest, servletResponse);
                        }

                        @Override
                        public AsyncContext startAsync() throws IllegalStateException {
                            return startAsync(this, null);
                        }
                    };
            if (klass != null) {
                checkHandler(request, this, klass);
            }
            servlet.service(request, this);
            checker.accept(this);
        }};
    }

    private void put(String name) {
        testHandler(
                DEFAULT_AUTH,
                PutHandler.class,
                name,
                q -> q.setContent(new byte[]{0xD, 0xE, 0xA, 0xD, 0xB, 0xE, 0xE, 0xF}),
                s -> Assert.assertContains(Cf.list(200, 201), s.getStatus())
        );
        mpfsClient.getFileInfoByUidAndPath(HandlersTest.DEFAULT_AUTH, name, Cf.list());
    }

    @Test
    public void testWebdavServlet() {
        Assert.assertThrows(servlet::getDavSessionProvider, UnsupportedOperationException.class);
        Assert.assertThrows(() -> servlet.setDavSessionProvider(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> servlet.setResourceFactory(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> servlet.setLocatorFactory(null), UnsupportedOperationException.class);
    }

    @Test
    @HttpRecorderRule.IgnoreParams
    @HttpRecorderRule.IgnoreHeaders
    public void testPutHandler() {
        put("/disk/" + Math.random());
    }

    @Test
    @HttpRecorderRule.IgnoreParams
    @HttpRecorderRule.IgnoreHeaders
    public void testPutHandlerWithRedirect() {
        String name = "/disk/" + Math.random();
        testHandler(
                auth(new ClientCapabilities(false, true, true, false, "/disk", false)),
                PutHandler.class,
                name,
                q -> q.setContent(new byte[]{0xD, 0xE, 0xA, 0xD, 0xB, 0xE, 0xE, 0xF}),
                s -> Assert.assertContains(Cf.list(302), s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreParams
    @HttpRecorderRule.IgnoreHeaders
    public void testPutHandlerWithConflict() {
        String name = "/photostream/2016-11-28 19-13-49.JPG";
        testHandler(
                auth(new ClientCapabilities(false, false, false, false, name, false)),
                PutHandler.class,
                name,
                q -> q.setContent(new byte[]{0xD, 0xE, 0xA, 0xD, 0xB, 0xE, 0xE, 0xF}),
                s -> {
                    Assert.assertContains(Cf.list(409), s.getStatus());
                    Assert.assertContains((String) s.getHeaderValue("Location"), "2016-11-28 19-13-49.JPG");
                }
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    public void testIndexHandler() {
        testHandler(
                DEFAULT_AUTH,
                IndexHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.addParameter("index", "!");
                },
                s -> Assert.assertEquals(200, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    public void testIndexHandler429() {
        testHandler(
                new AuthInfo(
                        Option.of(TestUser.uid),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        false,
                        Option.empty(),
                        Option.of(new OurClient(
                                "windows",
                                Option.of("1.4.22.5000"),
                                Option.of("autotests"),
                                Option.of("autotests")
                        )),
                        ClientCapabilities.getDefault(),
                        Option.empty(),
                        Option.empty(),
                        AuthInfo.AuthType.BASIC,
                        false
                ),
                IndexHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.addParameter("index", "!");
                },
                s -> Assert.assertEquals(500, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    public void testIndexHandler429NewSoftware() {
        testHandler(
                new AuthInfo(
                        Option.of(TestUser.uid),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        false,
                        Option.empty(),
                        Option.of(new OurClient(
                                "windows",
                                Option.of("1.4.22.5513"),
                                Option.of("autotests"),
                                Option.of("autotests")
                        )),
                        ClientCapabilities.getDefault(),
                        Option.empty(),
                        Option.of(false),
                        AuthInfo.AuthType.BASIC,
                        false
                ),
                IndexHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.addParameter("index", "!");
                },
                s -> Assert.assertEquals(429, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreParams
    @HttpRecorderRule.IgnoreHeaders
    public void testHeadHandler() {
        String name = "/disk/" + Math.random();
        put(name);
        testHandler(
                DEFAULT_AUTH,
                HeadHandler.class,
                name,
                q -> {
                },
                s -> Assert.assertEquals(200, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    public void testUserinfoHandler() {
        testHandler(
                new AuthInfo(
                        Option.of(TestUser.uid),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        false,
                        Option.empty(),
                        Option.of(new OurClient(
                                "mac",
                                Option.of("1.99999"),
                                Option.of("autotests"),
                                Option.of("autotests")
                        )),
                        ClientCapabilities.getDefault(),
                        Option.empty(),
                        Option.empty(),
                        AuthInfo.AuthType.OAUTH,
                        false
                ),
                UserinfoHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.addParameter("userinfo", "!");
                    q.addHeader("Yandex-OEM", "!");
                    q.addHeader("X-Install-DeviceID", "!");
                },
                s -> Assert.assertEquals(200, s.getStatus())
        );
        testHandler(
                new AuthInfo(
                        Option.of(TestUser.uid),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        false,
                        Option.empty(),
                        Option.of(OurClient.AUTOTESTS),
                        ClientCapabilities.getDefault(),
                        Option.empty(),
                        Option.empty(),
                        AuthInfo.AuthType.NONE,
                        false
                ),
                UserinfoHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.addParameter("userinfo", "!");
                },
                s -> Assert.assertEquals(401, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    @HttpRecorderRule.IgnoreParams
    public void testGetDigestHandler() {
        String name = "/" + Math.random();
        testHandler(
                DEFAULT_AUTH,
                PutHandler.class,
                name,
                q1 -> q1.setContent(new byte[]{0xD, 0xE, 0xA, 0xD, 0xB, 0xE, 0xE, 0xF}),
                s1 -> Assert.assertContains(Cf.list(200, 201), s1.getStatus())
        );
        testHandler(
                DEFAULT_AUTH,
                GetDigestHandler.class,
                name,
                q -> {
                    q.setQueryString("digest");
                    q.setPathInfo(name);
                },
                s -> Assert.assertContains(Cf.list(200), s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreParams
    @HttpRecorderRule.IgnoreHeaders
    public void testGetHandler() {
        String name = "/disk/" + Math.random();
        testHandler(
                DEFAULT_AUTH,
                PutHandler.class,
                name,
                q1 -> q1.setContent(new byte[]{0xD, 0xE, 0xA, 0xD, 0xB, 0xE, 0xE, 0xF}),
                s1 -> Assert.assertContains(Cf.list(200, 201), s1.getStatus())
        );
        testHandler(
                DEFAULT_AUTH,
                GetHandler.class,
                name,
                q -> q.setPathInfo(name),
                s -> Assert.assertEquals(200, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    @HttpRecorderRule.IgnoreParams
    public void testHeadForUploadHandler() {
        String name = "/" + Math.random();
        testHandler(
                DEFAULT_AUTH,
                HeadForUploadHandler.class,
                name,
                q -> {
                    q.addHeader("ETag", "wtf");
                    q.addHeader("Size", "42");
                    q.setPathInfo(name);
                },
                s -> Assert.assertEquals(404, s.getStatus())
        );
    }

    @Test
    public void testInstallHandler() {
        testHandler(
                DEFAULT_AUTH,
                InstallHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.addParameter("install", "!");
                },
                s -> Assert.assertEquals(404, s.getStatus())
        );
    }

    @Test
    public void testUnlockHandler() {
        testHandler(
                DEFAULT_AUTH,
                UnlockHandler.class,
                "/",
                q -> q.setPathInfo("/"),
                s -> Assert.assertEquals(204, s.getStatus())
        );
    }

    @Test
    public void testLockHandler() {
        testHandler(
                DEFAULT_AUTH,
                LockHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.setContent(("<?xml version=\"1.0\" encoding=\"utf-8\" ?>\n"
                            + "  <D:lockinfo xmlns:D=\"DAV:\">\n"
                            + "    <D:lockscope>\n"
                            + "      <D:exclusive/>\n"
                            + "    </D:lockscope>\n"
                            + "    <D:locktype>\n"
                            + "      <D:write/>\n"
                            + "    </D:locktype>\n"
                            + "  </D:lockinfo>").getBytes());
                },
                s -> Assert.assertEquals(200, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    @HttpRecorderRule.IgnoreParams
    public void testMkColHandler() {
        testHandler(
                DEFAULT_AUTH,
                MkColHandler.class,
                "/" + Math.random(),
                q -> q.setPathInfo("/"),
                s -> Assert.assertEquals(201, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    @HttpRecorderRule.IgnoreParams
    public void testMkColHandlerSysdir() {
        testHandler(
                DEFAULT_AUTH,
                MkColHandler.class,
                "/" + Math.random(),
                q -> {
                    q.setPathInfo("/");
                    q.setQueryString("default");
                },
                s -> Assert.assertEquals(500, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    public void testOperationsHandler() {
        testHandler(
                DEFAULT_AUTH,
                OperationsHandler.class,
                "/",
                q -> q.setPathInfo("/operations"),
                s -> Assert.assertEquals(207, s.getStatus())
        );
    }

    @Test
    public void testOptionsHandler() {
        testHandler(
                DEFAULT_AUTH,
                OptionsHandler.class,
                "/",
                q -> q.setPathInfo("/"),
                s -> Assert.assertEquals(200, s.getStatus())
        );
    }

    @Test
    public void testPingServlet() {
        testHandler(
                DEFAULT_AUTH,
                null,
                "/",
                q -> {
                    q.setPathInfo("/ping");
                    q.addParameter("ping", "!");
                    q.addParameter("delay", "42");
                },
                s -> Assert.assertEquals(200, s.getStatus())
        );
    }

    @Test
    public void testNotFoundServlet() {
        MockHttpServletResponse response = new MockHttpServletResponse();
        new NotFoundServlet().doGet(null, response);
        Assert.A.equals(404, response.getStatus());
    }

    @Test
    public void testPingHandler() {
        testHandler(
                new AuthInfo(
                        Option.of(TestUser.uid),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        Option.empty(),
                        false,
                        Option.empty(),
                        Option.empty(),
                        ClientCapabilities.getDefault(),
                        Option.empty(),
                        Option.empty(),
                        AuthInfo.AuthType.OAUTH,
                        false
                ),
                PingHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.addHeader("X-Yandex-Sdk-Version", "1.2.3");
                    q.addParameter("ping", "!");
                },
                s -> Assert.assertEquals(502, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    public void testPropfindHandler() {
        testHandler(
                DEFAULT_AUTH,
                PropfindHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.addParameter("count", "501");
                    q.setContent(("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
                            + "<D:propfind xmlns:D=\"DAV:\" xmlns:X=\"http://ns.example.com/foobar/\">\n"
                            + "  <D:prop>\n"
                            + "    <D:resourcetype/>\n"
                            + "    <X:foobar/>\n"
                            + "  </D:prop>\n"
                            + "</D:propfind>").getBytes());
                },
                s -> Assert.assertEquals(207, s.getStatus())
        );
        testHandler(
                DEFAULT_AUTH,
                PropfindHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.setContent(("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
                            + "<D:propfind xmlns:D=\"DAV:\" xmlns:X=\"http://ns.example.com/foobar/\">\n"
                            + "  <D:propname>\n"
                            + "    <D:resourcetype/>\n"
                            + "    <X:foobar/>\n"
                            + "  </D:propname>\n"
                            + "</D:propfind>").getBytes());
                },
                s -> Assert.assertEquals(207, s.getStatus())
        );
        testHandler(
                DEFAULT_AUTH,
                PropfindHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.setContent(("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
                            + "<D:propfind xmlns:D=\"DAV:\" xmlns:X=\"http://ns.example.com/foobar/\">\n"
                            + "  <D:allprop>\n"
                            + "    <D:resourcetype/>\n"
                            + "    <X:foobar/>\n"
                            + "  </D:allprop>\n"
                            + "</D:propfind>").getBytes());
                },
                s -> Assert.assertEquals(207, s.getStatus())
        );
        testHandler(
                DEFAULT_AUTH,
                PropfindHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.setContent(("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
                            + "<D:propfind xmlns:D=\"DAV:\" xmlns:X=\"http://ns.example.com/foobar/\">\n"
                            + "  <D:include>\n"
                            + "    <D:resourcetype/>\n"
                            + "    <X:foobar/>\n"
                            + "  </D:include>\n"
                            + "</D:propfind>").getBytes());
                },
                s -> Assert.assertEquals(207, s.getStatus())
        );
        testHandler(
                DEFAULT_AUTH,
                PropfindHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.setContent(("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
                            + "<D:propfind xmlns:D=\"DAV:\" xmlns:X=\"http://ns.example.com/foobar/\">\n"
                            + "  <D:multiple>\n"
                            + "  <resource path='x'/>\n"
                            + "  <resource path='y'/>\n"
                            + "  </D:multiple>\n"
                            + "</D:propfind>").getBytes());
                },
                s -> Assert.assertEquals(207, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    public void testPropfindHandlerStreaming() {
        testHandler(
                DEFAULT_AUTH,
                PropfindHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.addParameter("amount", "515");
                    q.addHeader(DavConstants.HEADER_DEPTH, "1");
                    q.setContent(("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
                            + "<D:propfind xmlns:D=\"DAV:\" xmlns:X=\"http://ns.example.com/foobar/\">\n"
                            + "  <D:prop>\n"
                            + "    <D:resourcetype/>\n"
                            + "    <X:foobar/>\n"
                            + "  </D:prop>\n"
                            + "</D:propfind>").getBytes());
                },
                s -> Assert.assertEquals(207, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    public void testProppatchHandler() {
        testHandler(
                DEFAULT_AUTH,
                ProppatchHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.setContent(("<?xml version=\"1.0\"?>\n"
                            +
                            "<d:propertyupdate xmlns:d=\"DAV:\" xmlns:o=\"urn:schemas-microsoft-com:office:office\">\n"
                            + "  <d:set>\n"
                            + "    <d:prop>\n"
                            + "      <o:Author>Douglas Groncki</o:Author>\n"
                            + "    </d:prop>\n"
                            + "  </d:set>\n"
                            + "</d:propertyupdate>").getBytes());
                },
                s -> Assert.assertEquals(409, s.getStatus())
        );
    }

    @Test(expected = DavException.class)
    @HttpRecorderRule.IgnoreHeaders
    public void testGetPreviewHandler() {
        testHandler(
                DEFAULT_AUTH,
                GetPreviewHandler.class,
                "/qwe",
                q -> {
                    q.setPathInfo("/qwe");
                    q.setQueryString("preview");
                    q.addParameter("size", "4x2");
                },
                s -> Assert.assertEquals(404, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    @HttpRecorderRule.IgnoreParams
    public void testPushHandler() {
        testHandler(
                DEFAULT_AUTH,
                PushHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.addParameter("push", "update");
                    q.addParameter("token", "42");
                    q.setContent(new byte[0]);
                },
                s -> Assert.assertEquals(400, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    @HttpRecorderRule.IgnoreParams
    public void testPushHandlerLimitExceeded() {
        testHandler(
                DEFAULT_AUTH,
                PushHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.addParameter("push", "update");
                    q.addParameter("token", "42");
                    q.setContent(new byte[0]);
                },
                s -> Assert.assertEquals(503, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    @HttpRecorderRule.IgnoreParams
    public void testPublishHandler() {
        String name = "/" + Math.random();
        testHandler(
                DEFAULT_AUTH,
                PublishHandler.class,
                name,
                q -> {
                    q.setPathInfo("/");
                    q.setQueryString("publish");
                },
                s -> Assert.assertEquals(404, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreHeaders
    @HttpRecorderRule.IgnoreParams
    public void testPublishHandlerUnpublish() {
        String name = "/" + Math.random();
        testHandler(
                DEFAULT_AUTH,
                PublishHandler.class,
                name,
                q -> {
                    q.setPathInfo("/");
                    q.setQueryString("unpublish");
                },
                s -> Assert.assertEquals(404, s.getStatus())
        );
    }

    @Test
    public void testRedirectToHelpHandler() {
        testHandler(
                DEFAULT_AUTH,
                RedirectToHelpHandler.class,
                "/",
                q -> q.setPathInfo("/"),
                s -> Assert.assertEquals(302, s.getStatus())
        );
    }

    @Test(expected = DavException.class)
    @HttpRecorderRule.IgnoreHeaders
    public void testAcceptInviteHandler() {
        testHandler(
                DEFAULT_AUTH,
                AcceptInviteHandler.class,
                "/",
                q -> {
                    q.setPathInfo("/");
                    q.setQueryString("share/not_approved/hash");
                },
                s -> Assert.assertEquals(302, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreParams
    @HttpRecorderRule.IgnoreHeaders
    public void testCopyHandler() {
        testHandler(
                DEFAULT_AUTH,
                CopyHandler.class,
                "/a",
                q -> {
                    q.setPathInfo("/a");
                    q.addHeader(DavConstants.HEADER_DESTINATION, "/b");
                },
                s -> Assert.assertEquals(404, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreParams
    @HttpRecorderRule.IgnoreHeaders
    public void testMoveHandler() {
        testHandler(
                DEFAULT_AUTH,
                MoveHandler.class,
                "/a",
                q -> {
                    q.setPathInfo("/a");
                    q.addHeader(DavConstants.HEADER_DESTINATION, "/b");
                },
                s -> Assert.assertEquals(404, s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreParams
    @HttpRecorderRule.IgnoreHeaders
    public void testTrashCleanHandler() {
        testHandler(
                ROOT_AUTH,
                TrashCleanHandler.class,
                "/trash",
                q -> {
                    q.setPathInfo("/trash");
                    q.setQueryString("clean");
                },
                s -> Assert.assertContains(Cf.list(200, 423), s.getStatus())
        );
    }

    @Test
    @HttpRecorderRule.IgnoreParams
    @HttpRecorderRule.IgnoreHeaders
    public void testDeleteHandler() {
        put("/disk/foo");
        testHandler(
                DEFAULT_AUTH,
                DeleteHandler.class,
                "/disk/foo",
                q -> q.setPathInfo("/disk/foo"),
                s -> Assert.assertEquals(200, s.getStatus())
        );
    }

}
