package ru.yandex.chemodan.app.webdav.servlet;

import java.io.IOException;
import java.net.UnknownHostException;

import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.webdav.repository.MpfsResource;
import ru.yandex.chemodan.app.webdav.repository.upload.UploaderClient;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsStoreOperation;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.io.http.HttpHeaderNames;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
public class HeadForUploadHandler implements DavMethodHandler {

    private static final Logger logger = LoggerFactory.getLogger(HeadForUploadHandler.class);

    private final MpfsClient mpfsClient;
    private final UploaderClient uploaderClient;

    public HeadForUploadHandler(MpfsClient mpfsClient, UploaderClient uploaderClient) {
        this.mpfsClient = mpfsClient;
        this.uploaderClient = uploaderClient;
    }

    @Override
    public void handle(WebdavRequest request, WebdavResponse response, MpfsResource resource)
            throws IOException, DavException
    {
        String etag = getEtag(request).get();
        Option<Long> size = HttpServletRequestX.wrap(request).getHeaderO("Size").flatMapO(Cf.Long::parseSafe);

        if (!size.isPresent()) {
            throw new DavException(HttpStatus.SC_404_NOT_FOUND);
        }

        Tuple2List<String, String> headers = Tuple2List.tuple2List();
        Option<String> userAgent = HttpServletRequestX.wrap(request).getUserAgent();
        if (userAgent.isPresent()) {
            headers = headers.plus1(HttpHeaderNames.USER_AGENT, userAgent.get());
        }

        MpfsStoreOperation operation = mpfsClient.astore(resource.getUser(), resource.getRealPath(), etag, headers);
        //TODO: check right operation type

        String uploadUrl = operation.getUploadUrl().getOrThrow(() -> new DavException(HttpStatus.SC_404_NOT_FOUND));

        if (!uploadUrl.startsWith("https")) {
            if (uploadUrl.contains(":32480/")) {
                uploadUrl = uploadUrl.replaceFirst("http://", "https://").replace(":32480", "");
            } else {
                logger.error(new IllegalStateException("External urls must start with https: " + uploadUrl));
            }
        }

        try {
            response.addHeader("Content-Length", Long.toString(uploaderClient.getUploadedSize(uploadUrl)));
            response.addHeader("Location", uploadUrl);
        } catch (RuntimeIoException e) {
            if (e.getCause() instanceof UnknownHostException) {
                // removed old uploader
                // java.net.UnknownHostException: uploader9m.disk.yandex.net
                throw new DavException(HttpStatus.SC_404_NOT_FOUND);
            }
            throw e;
        }
    }

    @Override
    public String method() {
        return "HEAD";
    }

    @Override
    public boolean matches(WebdavRequest request, MpfsResource resource) {
        return getEtag(request).isPresent();
    }

    private Option<String> getEtag(WebdavRequest request) {
        HttpServletRequestX reqX = HttpServletRequestX.wrap(request);

        return reqX.getHeaderO("Yandex-Diff").orElse(reqX.getHeaderO("ETag"));
    }

    @Override
    public int order() {
        return 2;
    }
}
