package ru.yandex.chemodan.app.webdav.servlet;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import org.apache.http.HttpResponse;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.auth.OurClient;
import ru.yandex.chemodan.app.webdav.repository.MpfsResource;
import ru.yandex.chemodan.app.webdav.servlet.index.MpfsIndexWriter;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.util.http.RequestUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
public class IndexHandler implements DavMethodHandler {
    private final MpfsClient mpfsClient;


    public IndexHandler(MpfsClient mpfsClient) {
        this.mpfsClient = mpfsClient;
    }

    @Override
    public void handle(WebdavRequest request, WebdavResponse servletResponse, MpfsResource resource)
            throws IOException, DavException
    {
        Option<String> ifMatchStr = RequestUtils.getIfMatch(request);
        Option<Long> ifMatch = ifMatchStr.flatMapO(Cf.Long::parseSafe);

        if (ifMatchStr.isPresent() && !ifMatch.isPresent()) {
            throw new DavException(HttpStatus.SC_404_NOT_FOUND);
        }

        String path = resource.getRealPath();

        Option<HttpException> httpException = mpfsClient.diff(resource.getUser(), path, ifMatch, (response) -> {
            try {
                Option<HttpException> mpfsException = checkMpfsError(request, response);
                if (mpfsException.isPresent()) {
                    if (response.getStatusLine().getStatusCode() == HttpStatus.SC_429_TOO_MANY_REQUESTS) {
                        return mpfsException;
                    } else {
                        throw mpfsException.get();
                    }
                }

                long version = Cf.Long.parse(response.getFirstHeader("version").getValue());
                long itemsCount = Cf.Long.parse(response.getFirstHeader("Item-Count").getValue());
                long size = Cf.Long.parse(response.getFirstHeader("Content-Length").getValue());

                if (ifMatch.isSome(version)) {
                    //nothing changed
                    servletResponse.setStatus(HttpStatus.SC_200_OK);
                    servletResponse.addHeader("ETag", ifMatch.get().toString());
                    servletResponse.addHeader("Item-Count", itemsCount + "");
                    servletResponse.flushBuffer();
                    return Option.empty();
                }

                servletResponse.addHeader("ETag", version + "");
                servletResponse.addHeader("Item-Count", itemsCount + "");

                JsonParser jsonParser = (new JsonFactory()).createParser(response.getEntity().getContent());

                HttpServletRequestX reqX = HttpServletRequestX.wrap(request);
                MpfsIndexWriter writer = new MpfsIndexWriter(jsonParser,
                        reqX.getParameterO("ext").flatMap(p -> Cf.list(StringUtils.split(p, ","))),
                        reqX.getParameterO("v").getOrElse("")
                );

                writer.writeBinaryIndex(servletResponse.getOutputStream());
                servletResponse.flushBuffer();
            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
            return Option.empty();
        });

        httpException.forEach(ExceptionUtils::throwException);
    }

    @Override
    public String method() {
        return "GET";
    }

    @Override
    public boolean matches(WebdavRequest request, MpfsResource resource) {
        return request.getParameter("index") != null;
    }

    @Override
    public int order() {
        return 2;
    }

    private boolean isClientVersionOlder(OurClient ourClient, String os, Integer majorVersion, Integer build) {
        return ourClient.os.equals(os) &&
               ourClient.majorVersion.isSome(majorVersion) &&
               ourClient.build.isMatch(b -> b < build);
    }

    private Option<HttpException> checkMpfsError(WebdavRequest request, HttpResponse mpfsResponse) {
        int statusCode = mpfsResponse.getStatusLine().getStatusCode();
        if (HttpStatus.is2xx(statusCode)) {
            return Option.empty();
        }
        if (statusCode == HttpStatus.SC_429_TOO_MANY_REQUESTS) {
            // Old clients cannot handle 429 error. https://st.yandex-team.ru/CHEMODAN-58713
            AuthInfo authInfo = getAuthInfo(request);
            if (authInfo.isOurClient()) {
                OurClient ourClient = authInfo.ourClient.get();
                if (isClientVersionOlder(ourClient, "mac", 1, 6271) ||
                    isClientVersionOlder(ourClient, "mac", 3, 1612) ||
                    isClientVersionOlder(ourClient, "windows", 1, 5513) ||
                    isClientVersionOlder(ourClient, "windows", 3, 2535)
                ) {
                    statusCode = HttpStatus.SC_503_SERVICE_UNAVAILABLE;
                }
            }
        }

        return Option.of(new HttpException(statusCode, Abstract200ExtendedResponseHandler.getErrorMessage(mpfsResponse)));
    }
}
