package ru.yandex.chemodan.app.webdav.servlet;

import java.io.IOException;
import java.io.InputStream;

import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.webdav.repository.PathUtils;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.xml.dom.DomUtils;

/**
 * @author tolmalev
 */
public class MultipleRequest {
    private static final String MULTIPLE_HEADER = "X-Yandex-Multiple";
    static final String TAG_NAME = "multiple";

    public final ListF<String> paths;

    public MultipleRequest(ListF<String> paths) {
        Validate.notEmpty(paths);
        this.paths = paths;
    }

    public static Option<MultipleRequest> parse(WebdavRequest request) throws DavException {
        // todo: check lowercase header
        if (Option.ofNullable(request.getHeader(MULTIPLE_HEADER)).getOrElse("").equals("")) {
            return Option.empty();
        }
        try {
            return Option.of(parse(request.getInputStream()));
        } catch (IOException | RuntimeException e) {
            throw new DavException(HttpStatus.SC_400_BAD_REQUEST, "bad multiple request");
        }
    }

    public static MultipleRequest parse(InputStream in) throws DavException {
        try {
            Document document = DomUtils.read(in);
            return parseMultipleDocument(document);
        } catch (RuntimeException e) {
            throw new DavException(HttpStatus.SC_400_BAD_REQUEST, e);
        }
    }

    private static MultipleRequest parseMultipleDocument(Document document) throws DavException {
        NodeList multiples = document.getElementsByTagName(TAG_NAME);
        if (multiples.getLength() != 1) {
            throw new DavException(HttpStatus.SC_400_BAD_REQUEST, "bad xml");
        }

        Node multiple = multiples.item(0);
        return parseMultipleNode(multiple);
    }

    static MultipleRequest parseMultipleNode(Node multiple) throws DavException {
        NodeList childNodes = multiple.getChildNodes();

        ListF<String> paths = Cf.arrayList();

        for (int i = 0; i < childNodes.getLength(); i++) {
            Node item = childNodes.item(i);
            if ("resource".equals(item.getNodeName())) {
                Node path = item.getAttributes().getNamedItem("path");
                if (path != null) {
                    paths.add(path.getTextContent());
                } else {
                    paths.add(PathUtils.decodePath(item.getTextContent().trim()));
                }
            }
        }

        if (paths.isEmpty()) {
            throw new DavException(HttpStatus.SC_400_BAD_REQUEST, "no resources");
        }
        return new MultipleRequest(paths);
    }
}
