package ru.yandex.chemodan.app.webdav.servlet;

import java.io.IOException;
import java.util.Date;

import com.fasterxml.jackson.databind.util.ISO8601Utils;
import org.apache.http.client.utils.DateUtils;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.MultiStatus;
import org.apache.jackrabbit.webdav.MultiStatusResponse;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;
import org.apache.jackrabbit.webdav.property.DefaultDavProperty;
import org.apache.jackrabbit.webdav.property.ResourceType;
import org.apache.jackrabbit.webdav.xml.Namespace;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.webdav.repository.MpfsResource;
import ru.yandex.chemodan.app.webdav.repository.properties.DavProperties;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsOperation;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author tolmalev
 */
public class OperationsHandler implements DavMethodHandler {
    private static final Namespace NAMESPACE = Namespace.getNamespace(null, "urn:yandex:disk:operations");

    private final MpfsClient mpfsClient;

    public OperationsHandler(MpfsClient mpfsClient) {
        this.mpfsClient = mpfsClient;
    }

    @Override
    public void handle(WebdavRequest request, WebdavResponse response, MpfsResource resource)
            throws IOException, DavException
    {
        MultiStatus mstatus = new MultiStatus();

        MultiStatusResponse rootOperations = new MultiStatusResponse("/operations", null);

        rootOperations.add(new ResourceType(ResourceType.COLLECTION));
        rootOperations.add(new DefaultDavProperty<>(
                "getlastmodified", "Sat, 01 Jan 2000 12:00:00 GMT", DavProperties.DAV));
        rootOperations.add(new DefaultDavProperty<>(
                "creationdate", "2000-01-01T12:00:00Z", DavProperties.DAV));

        mstatus.addResponse(rootOperations);

        mpfsClient
                .activeOperations(resource.getUser())
                .flatMap(this::consOperationResponse)
                .forEach(mstatus::addResponse);

        response.sendMultiStatus(mstatus);
    }

    private Option<MultiStatusResponse> consOperationResponse(MpfsOperation operation) {
        return getOpMethod(operation).map(method -> {
            MultiStatusResponse response = new MultiStatusResponse("/?op_status&id=" + operation.getOid(), null);

            response.add(new ResourceType(ResourceType.DEFAULT_RESOURCE));
            response.add(new DefaultDavProperty<>("method", method, NAMESPACE));

            operation.getMtime().forEach(mtime -> response.add(new DefaultDavProperty<>(
                    "getlastmodified", DateUtils.formatDate(new Date(mtime * 1000)), DavProperties.DAV)));

            operation.getCtime().forEach(ctime -> response.add(new DefaultDavProperty<>(
                    "creationdate", ISO8601Utils.format(new Date(ctime * 1000)), DavProperties.DAV)));

            operation.getData().ifPresent(data -> {
                if (method.equals("COPY") || method.equals("MOVE")) {
                    addProperty(data, "source", response, "path");
                    addProperty(data, "target", response, "destination");
                } else if (operation.getType().isSome("trash") && operation.getSubtype().isSome("restore")) {
                    addProperty(data, "target", response, "path");
                } else if (operation.getType().isSome("trash") && operation.getSubtype().isSome("drop")) {
                    response.add(new DefaultDavProperty<>("path", "/trash/", NAMESPACE));
                } else {
                    addProperty(data, "path", response, "path");
                }
            });

            return response;
        });
    }

    private void addProperty(BenderJsonNode data, String field, MultiStatusResponse response, String property) {
        data.getField(field).forEach(source -> response.add(new DefaultDavProperty<>(property, stripPath(source.getString()), NAMESPACE)));
    }

    private String stripPath(String path) {
        return Cf.x(path.split(":", 2)).last();
    }

    private static final MapF<String, String> TYPES = Cf
            .map("copy", "COPY")
            .plus1("move", "MOVE")
            .plus1("office", "LOCK")
            .plus1("remove", "DELETE")
            .unmodifiable();

    private static final MapF<String, String> SUBTYPES = Cf
            .map("append", "DELETE")
            .plus1("restore", "MKCOL")
            .plus1("droppath", "DELETE")
            .plus1("drop", "DELETE")
            .unmodifiable();

    private static Option<String> getOpMethod(MpfsOperation operation) {
        if (operation.getType().isSome("trash")) {
            return operation.getSubtype().flatMapO(SUBTYPES::getO);
        } else {
            return operation.getType().flatMapO(TYPES::getO);
        }
    }

    @Override
    public String method() {
        return "PROPFIND";
    }

    @Override
    public boolean matches(WebdavRequest request, MpfsResource resource) {
        return "operations".equals(StringUtils.removeStart(request.getPathInfo(), "/"));
    }

    @Override
    public int order() {
        return 2;
    }
}
