package ru.yandex.chemodan.app.webdav.servlet;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Stream;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.stream.XMLStreamWriter;

import com.ctc.wstx.stax.WstxOutputFactory;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.apache.commons.io.IOUtils;
import org.apache.jackrabbit.webdav.DavConstants;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.DavServletResponse;
import org.apache.jackrabbit.webdav.MultiStatus;
import org.apache.jackrabbit.webdav.MultiStatusResponse;
import org.apache.jackrabbit.webdav.Status;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;
import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DavPropertyNameSet;
import org.apache.jackrabbit.webdav.property.PropContainer;
import org.apache.jackrabbit.webdav.xml.DomUtil;
import org.apache.jackrabbit.webdav.xml.ElementIterator;
import org.jetbrains.annotations.NotNull;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.repository.MpfsResource;
import ru.yandex.chemodan.app.webdav.repository.MpfsResourceManager;
import ru.yandex.chemodan.app.webdav.repository.PathUtils;
import ru.yandex.chemodan.app.webdav.repository.properties.DavProperties;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.misc.xml.dom.DomUtils;
import ru.yandex.misc.xml.stream.XmlWriter;

import static org.apache.jackrabbit.webdav.DavConstants.NAMESPACE;
import static org.apache.jackrabbit.webdav.DavConstants.PROPFIND_ALL_PROP;
import static org.apache.jackrabbit.webdav.DavConstants.PROPFIND_ALL_PROP_INCLUDE;
import static org.apache.jackrabbit.webdav.DavConstants.PROPFIND_BY_PROPERTY;
import static org.apache.jackrabbit.webdav.DavConstants.PROPFIND_PROPERTY_NAMES;
import static org.apache.jackrabbit.webdav.DavConstants.XML_ALLPROP;
import static org.apache.jackrabbit.webdav.DavConstants.XML_INCLUDE;
import static org.apache.jackrabbit.webdav.DavConstants.XML_PROP;
import static org.apache.jackrabbit.webdav.DavConstants.XML_PROPFIND;
import static org.apache.jackrabbit.webdav.DavConstants.XML_PROPNAME;

/**
 * @author tolmalev
 */
@RequiredArgsConstructor
public class PropfindHandler implements DavMethodHandler {
    private static final Logger log = LoggerFactory.getLogger(PropfindHandler.class);

    private final MpfsResourceManager manager;
    private final ThreadLocal<String> requestDump = new ThreadLocal<>();

    @Override
    public void handle(WebdavRequest request, WebdavResponse response, MpfsResource resource)
            throws DavException, IOException
    {
        requestDump.remove();

        int depth = getDepth(request);
        AuthInfo authInfo = getAuthInfo(request);
        PropfindRequest propfind = parsePropfind(request);
        HttpServletRequestX reqX = HttpServletRequestX.wrap(request);
        ListF<MpfsResource> resources = overrideResource(authInfo, resource, reqX, propfind);

        //streaming via mpfsClient.streamingListByUidAndPath
        if (resources.size() == 1 && resources.single().supportsStreamingPropfind() && depth > 0 &&
                !reqX.getNonEmptyParameterO("amount").flatMapO(Cf.Integer::parseSafe).exists(amount -> amount <= 500))
        {
            makeStreamingPropfindResponse(authInfo, resources.single().getRealPath(), propfind, reqX, response);
            return;
        }

        if (resources.size() > 1) {
            //XXX: hack for multiple requests
            depth = 0;
        }

        Option<MpfsResource> currentResource = Option.empty();
        MultiStatus mstatus = new MultiStatus();
        try {
            for (MpfsResource propfindRes : resources) {
                currentResource = Option.of(propfindRes);
                mstatus.addResponse(consResponse(propfindRes, propfind.requestProperties, propfind.propfindType));
                if (depth > 0 && propfindRes.isCollection()) {
                    propfindRes.getMembers().forEachRemaining(
                            r -> mstatus.addResponse(
                                    consResponse((MpfsResource) r, propfind.requestProperties, propfind.propfindType)));
                }
            }
        } catch (Throwable t) {
            ExceptionUtils.throwIfUnrecoverable(t);
            if (log.isTraceEnabled()) {
                log.trace("Failed request body: " + requestDump.get());

                currentResource.forEach(
                        r -> log.trace("Failed resource real path: " + r.getRealPath() + " info: " + r.getFileInfo()));
            }
            throw t;
        }
        response.sendMultiStatus(mstatus);
    }

    private ListF<MpfsResource> overrideResource(AuthInfo authInfo, MpfsResource resource, HttpServletRequestX reqX,
            PropfindRequest propfind)
    {
        // add support of multiple paths
        // see https://wiki.yandex-team.ru/disk/webdav/
        DavPropertyNameSet requestedProperties = propfind.requestProperties;
        ListF<String> fullPaths = propfind
                .multiplePaths.flatMap(Function.identityF())
                .map(path -> PathUtils.getOneMultiplePath(path, resource.getRealPath()));

        if (!fullPaths.isEmpty()) {
            ListF<DavPropertyName> propertiesToFetch = getPropertiesToSelect(reqX, requestedProperties);
            return manager.createResources(authInfo, propertiesToFetch, fullPaths);
        } else {
            resource.addFetchProperties(requestedProperties.getContent());
            return Cf.list(resource);
        }
    }

    private ListF<DavPropertyName> getPropertiesToSelect(HttpServletRequestX reqX,
            DavPropertyNameSet requestedProperties)
    {
        return Cf.toArrayList(manager.getHackDefaultProperties(reqX)).plus(requestedProperties.getContent());
    }


    private int getDepth(WebdavRequest request) throws DavException {
        int depth;
        try {
            depth = request.getDepth(0);
        } catch (IllegalArgumentException e) {
            throw new DavException(HttpStatus.SC_403_FORBIDDEN);
        }
        depth = Math.min(depth, 1);
        return depth;
    }

    private void makeStreamingPropfindResponse(AuthInfo authInfo, String realPath, PropfindRequest propfind,
            HttpServletRequestX reqX, WebdavResponse response)
    {
        manager.streamingListing(authInfo, realPath,
                reqX.getNonEmptyParameterO("offset").flatMapO(Cf.Integer::parseSafe),
                reqX.getNonEmptyParameterO("amount").flatMapO(Cf.Integer::parseSafe),
                reqX.getNonEmptyParameterO("sort"),
                reqX.getNonEmptyParameterO("order").isSome("desc"),
                getPropertiesToSelect(reqX, propfind.requestProperties),
                (it) -> {
                    response.setStatus(HttpStatus.SC_207_MULTI_STATUS);
                    response.setContentType("application/xml; charset=UTF-8");

                    try (XmlWriter xw = createXmlWriter(response)) {
                        xw.startDocument("UTF-8", "1.0");

                        //MultiStatus
                        xw.startElement("d", DavConstants.XML_MULTISTATUS, DavConstants.NAMESPACE.getURI());

                        it.forEachRemaining(resource -> {
                            HackedMultiStatusResponse oneResponse =
                                    consResponse(resource, propfind.requestProperties, propfind.propfindType);
                            try {
                                writeNode(xw, oneResponse.toXml(DomUtil.createDocument()));
                            } catch (ParserConfigurationException e) {
                                throw ExceptionUtils.translate(e);
                            }
                        });

                        xw.endElement();
                        xw.endDocument();
                        xw.flush();
                    }
                }
        );
    }

    @NotNull
    @SneakyThrows
    private static XmlWriter createXmlWriter(WebdavResponse response) {
        WstxOutputFactory factory = new WstxOutputFactory() {{
            configureForSpeed();
            getConfig().doSupportNamespaces(true);
            getConfig().enableAutomaticNamespaces(true);
        }};
        return new XmlStreamWriterAdapter(factory.createXMLStreamWriter(response.getOutputStream()));
    }

    private static void writeNode(XmlWriter xw, Node node) {
        if (node instanceof Element) {
            Element element = (Element) node;
            if (node.getPrefix() == null && node.getNamespaceURI() == null) {
                xw.startElement(element.getLocalName());
            } else {
                xw.startElement(Option.ofNullable(node.getPrefix()).getOrElse(""), element.getLocalName(),
                        element.getNamespaceURI());
            }
            DomUtils.attributes(element)
                    .forEach(a -> xw.textElement(a.getPrefix(), a.getNamespaceURI(), a.getLocalName(), a.getValue()));
            DomUtils.childNodes(element).forEach(child -> writeNode(xw, child));
            xw.endElement();
        }

        if (node.getNodeType() == Node.TEXT_NODE) {
            xw.addCharacters(node.getTextContent());
        }
    }

    private HackedMultiStatusResponse consResponse(MpfsResource resource, DavPropertyNameSet propNameSet,
            int propFindType)
    {
        // only property names requested
        if (propFindType == MultiStatusResponse.PROPFIND_PROPERTY_NAMES) {
            HackedMultiStatusResponse response = new HackedMultiStatusResponse(resource.getHref());
            Stream.of(resource.getPropertyNames()).forEach(propName -> response.add(propName, 200));
            return response;
        }


        HackedMultiStatusResponse response = new HackedMultiStatusResponse(resource.getHref());
        // all or a specified set of property and their values requested.
        Set<DavPropertyName> requestedNames = new HashSet<>(propNameSet.getContent());

        // Add requested properties or all non-protected properties,
        // or non-protected properties plus requested properties (allprop/include)
        if (propFindType == MultiStatusResponse.PROPFIND_BY_PROPERTY) {
            // add explicitly requested properties (proptected or non-protected)
            for (DavPropertyName propName : propNameSet) {
                ListF<DavProperty> props = resource.getProperties(propName);
                if (props != null) {
                    props.forEach(p -> response.add(p, 200));
                    requestedNames.remove(propName);
                }
            }
        } else {
            // add all non-protected properties
            for (DavProperty<?> property : resource.getProperties()) {
                boolean wasRequested = requestedNames.remove(property.getName());

                if (!property.isInvisibleInAllprop() || wasRequested) {
                    response.add(property, 200);
                }
            }

            // try if missing properties specified in the include section
            // can be obtained using resource.getProperty
            if (propFindType == MultiStatusResponse.PROPFIND_ALL_PROP_INCLUDE && !requestedNames.isEmpty()) {
                for (DavPropertyName propName : new HashSet<>(requestedNames)) {
                    ListF<DavProperty> props = resource.getProperties(propName);
                    if (props != null) {
                        props.forEach(p -> response.add(p, 200));
                        requestedNames.remove(propName);
                    }
                }
            }
        }

        //all unprocessed requested props must be NOT FOUND
        if (!requestedNames.isEmpty()) {
            requestedNames.forEach(propName -> response.add(propName, 404));
        }

        return response;
    }

    private String getNameSpace(Node node) {
        if (node == null) {
            return "";
        } else if (node.getNamespaceURI() != null) {
            return node.getNamespaceURI();
        } else {
            return getNameSpace(node.getParentNode());
        }
    }

    //XXX: copy-paste from jackrabbit at most
    private PropfindRequest parsePropfind(WebdavRequest request) throws DavException {
        Option<ListF<String>> paths = Option.empty();
        DavPropertyNameSet propfindProps = new DavPropertyNameSet();
        int propfindType = PROPFIND_ALL_PROP;

        Document requestDocument = getRequestDocument(request);
        // propfind httpRequest with empty body >> retrieve all property
        if (requestDocument == null) {
            return new PropfindRequest(Option.empty(), propfindProps, propfindType);
        }

        // propfind httpRequest with invalid body
        Element root = requestDocument.getDocumentElement();
        if (!XML_PROPFIND.equals(root.getLocalName())) {
            log.info("PropFind-Request has no <propfind> tag.");
            throw new DavException(DavServletResponse.SC_BAD_REQUEST, "PropFind-Request has no <propfind> tag.");
        }

        DavPropertyNameSet include = null;

        ElementIterator it = DomUtil.getChildren(root);
        int propfindTypeFound = 0;

        while (it.hasNext()) {
            Element child = it.nextElement();
            String nodeName = child.getLocalName();
            if (NAMESPACE.getURI().equals(getNameSpace(child))) {
                if (XML_PROP.equals(nodeName)) {
                    propfindType = PROPFIND_BY_PROPERTY;
                    propfindProps = new DavPropertyNameSet(child);
                    propfindTypeFound += 1;
                } else if (XML_PROPNAME.equals(nodeName)) {
                    propfindType = PROPFIND_PROPERTY_NAMES;
                    propfindTypeFound += 1;
                } else if (XML_ALLPROP.equals(nodeName)) {
                    propfindType = PROPFIND_ALL_PROP;
                    propfindTypeFound += 1;
                } else if (XML_INCLUDE.equals(nodeName)) {
                    include = new DavPropertyNameSet() {{
                        DomUtil.getChildren(child).forEachRemaining(e -> add(DavPropertyName.createFromXml(e)));
                    }};
                } else if (MultipleRequest.TAG_NAME.equals(nodeName)) {
                    paths = Option.of(MultipleRequest.parseMultipleNode(child).paths);
                }
            }
            if (DavProperties.USER_STATES.getURI().equals(child.getNamespaceURI())) {
                if (XML_ALLPROP.equals(nodeName)) {
                    propfindType = PROPFIND_BY_PROPERTY;
                    propfindProps.add(DavPropertyName.create("allprop", DavProperties.USER_STATES));
                    propfindTypeFound += 1;
                }
            }
        }

        if (propfindTypeFound > 1) {
            log.info("Multiple top-level propfind instructions");
            throw new DavException(DavServletResponse.SC_BAD_REQUEST, "Multiple top-level propfind instructions");
        }

        if (include != null) {
            if (propfindType == PROPFIND_ALL_PROP) {
                // special case: allprop with include extension
                propfindType = PROPFIND_ALL_PROP_INCLUDE;
                propfindProps = include;
            } else {
                throw new DavException(DavServletResponse.SC_BAD_REQUEST, "<include> goes only with <allprop>");
            }
        }
        log.debug("Propfind type: {}, props: {}", propfindType, propfindProps.getContent());
        return new PropfindRequest(paths, propfindProps, propfindType);
    }

    private Document getRequestDocument(WebdavRequest httpRequest) throws DavException {
        /*
        Don't attempt to parse the body if the content length header is 0.
        NOTE: a value of -1 indicates that the length is unknown, thus we have
        to parse the body. Note that http1.1 request using chunked transfer
        coding will therefore not be detected here.
        */
        if (httpRequest.getContentLength() == 0) {
            return null;
        }

        // try to parse the request body
        try {
            InputStream in = httpRequest.getInputStream();
            String requestBody = IOUtils.toString(in, Charset.defaultCharset());
            requestDump.set(requestBody);
            in = IOUtils.toInputStream(requestBody, Charset.defaultCharset());

            // use a buffered input stream to find out whether there actually
            // is a request body
            InputStream bin = new BufferedInputStream(in);
            bin.mark(1);
            boolean isEmpty = -1 == bin.read();
            bin.reset();
            return isEmpty ? null : DomUtil.parseDocument(bin);
        } catch (IOException | SAXException e) {
            if (log.isDebugEnabled()) {
                log.debug("Unable to build an XML Document from the request body: " + e.getMessage());
            }
            throw new DavException(DavServletResponse.SC_BAD_REQUEST, e);
        } catch (ParserConfigurationException e) {
            if (log.isDebugEnabled()) {
                log.debug("Unable to build an XML Document from the request body: " + e.getMessage());
            }
            throw new DavException(DavServletResponse.SC_INTERNAL_SERVER_ERROR, e);
        }
    }

    @AllArgsConstructor(access = AccessLevel.PRIVATE)
    private static class PropfindRequest {
        final Option<ListF<String>> multiplePaths;
        final DavPropertyNameSet requestProperties;
        final int propfindType;
    }

    @Override
    public String method() {
        return "PROPFIND";
    }

    private static class HackedMultiStatusResponse extends MultiStatusResponse {

        HackedMultiStatusResponse(String href) {
            super(href, null);
        }

        @Override
        public Element toXml(Document document) {
            Element response = DomUtil.createElement(document, XML_RESPONSE, NAMESPACE);
            // add '<href>'
            response.appendChild(DomUtil.hrefToXml(getHref(), document));
            if (isPropStat()) {
                // add '<propstat>' elements
                Cf.x(getStatus()).sortedBy(Status::getStatusCode).forEach(st -> {
                    // statusMap.get(st.getStatusCode()) is private, so use hack
                    PropContainer propCont = getProperties(st.getStatusCode());
                    if (propCont.isEmpty()) {
                        propCont = getPropertyNames(st.getStatusCode());
                    }

                    if (!propCont.isEmpty()) {
                        Element propstat = DomUtil.createElement(document, XML_PROPSTAT, NAMESPACE);
                        propstat.appendChild(st.toXml(document));
                        propstat.appendChild(propCont.toXml(document));
                        response.appendChild(propstat);
                    }
                });
            } else {
                // add a single '<status>' element
                // NOTE: a href+status response cannot be created with 'null' status
                response.appendChild(Cf.x(getStatus()).single().toXml(document));
            }
            // add the optional '<responsedescription>' element
            String description = getResponseDescription();
            if (description != null) {
                Element desc = DomUtil.createElement(document, XML_RESPONSEDESCRIPTION, NAMESPACE);
                DomUtil.setText(desc, description);
                response.appendChild(desc);
            }
            return response;
        }
    }

    private static final class XmlStreamWriterAdapter extends ru.yandex.misc.xml.stream.XmlStreamWriterAdapter {

        XmlStreamWriterAdapter(XMLStreamWriter xmlStreamWriter) {
            super(xmlStreamWriter);
        }

        @Override
        protected void finalize() {
        }

        @Override
        protected CharSequence invalidCharsToSpaces(CharSequence string) {
            return string;
        }

        @Override
        protected char[] invalidCharsToSpaces(char[] chars, int start, int length) {
            return chars;
        }

    }
}
