package ru.yandex.chemodan.app.webdav.servlet;

import java.io.IOException;
import java.util.Set;

import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.auth.ClientCapabilities;
import ru.yandex.chemodan.app.webdav.repository.MpfsResource;
import ru.yandex.chemodan.app.webdav.repository.MpfsResourceManager;
import ru.yandex.chemodan.app.webdav.repository.upload.Upload;
import ru.yandex.chemodan.app.webdav.repository.upload.UploadUtils;
import ru.yandex.chemodan.app.webdav.repository.upload.UploaderClient;
import ru.yandex.chemodan.mpfs.MpfsResponseParserUtils;
import ru.yandex.chemodan.mpfs.MpfsStoreOperation;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
public class PutHandler extends DavHandlerBase {
    private static final Logger logger = LoggerFactory.getLogger(PutHandler.class);

    private static final Set<Integer> UPLOAD_TRAFFIC_LIMIT_MPFS_CODES = Cf.set(311, 312);

    private final DataSize redirectPutThreshold;
    private final UploaderClient uploaderClient;

    private final Pattern2 invisiblePattern;

    protected PutHandler(MpfsResourceManager manager, DataSize redirectPutThreshold, UploaderClient uploaderClient,
            Pattern2 invisiblePattern)
    {
        super(manager, "PUT");
        this.redirectPutThreshold = redirectPutThreshold;
        this.uploaderClient = uploaderClient;
        this.invisiblePattern = invisiblePattern;
    }

    @Override
    public void handle(WebdavRequest request, WebdavResponse response, MpfsResource resource)
            throws IOException, DavException
    {
        //TODO: right size (check for header X-Expected-Entity-Length)

        // deny PUT for collections
        if (StringUtils.endsWithIgnoreCase(request.getRequestLocator().getResourcePath(), "/")) {
            throw new DavException(HttpStatus.SC_405_METHOD_NOT_ALLOWED);
        }

        HttpServletRequestX reqX = HttpServletRequestX.wrap(request);
        Upload upload = UploadUtils.extractUploadFromRequest(reqX);
        boolean isRedirectUpload = isRedirectUpload(getAuthInfo(request).clientCapabilities, upload);

        Option<String> userAgent = reqX.getUserAgent();

        MpfsStoreOperation store =
                store(request, response, upload, resource, isRedirectUpload, userAgent, reqX.getRemoteIpAddress());

        if (store.getStatus().isSome("hardlinked")) {
            store.getDiskPath().forEach(location -> setLocationHeader(response, location));
            response.setStatus(201);
            response.getWriter().print("Hardlinked");
            return;
        }

        if (isRedirectUpload) {
            String redirectUrl = store.getUploadUrl().get();
            if (!redirectUrl.startsWith("https")) {
                throw new IllegalStateException("external urls must start with https: " + redirectUrl);
            }
            response.addHeader("Location", redirectUrl);
            response.addHeader("Redirect-Ref", redirectUrl);
            addLivePhotoOperationIdIfNecessary(response, upload, store);
            response.setStatus(302);
            logger.debug("Redirect for PUT: {}", redirectUrl);
        } else {
            PutResult putResult = uploaderClient.sendToUploader(reqX.getInputStream(), upload, store.getUploadUrl().get());
            addLivePhotoOperationIdIfNecessary(response, upload, store);
            response.setStatus(putResult.status);

            putResult.location.ifPresent(location -> setLocationHeader(response, location));
        }
    }

    private void addLivePhotoOperationIdIfNecessary(WebdavResponse response, Upload upload, MpfsStoreOperation store) {
        if (upload.livePhotoType.isSome("photo")) {
            response.addHeader("X-Yandex-Live-Photo-Operation", store.getOid());
        }
    }

    private MpfsStoreOperation store(WebdavRequest request, WebdavResponse response, Upload upload,
            MpfsResource resource, boolean isRedirectUpload,
            Option<String> userAgent, IpAddress remoteIp) throws DavException
    {
        try {
            return manager.store(
                    upload,
                    resource.getUser(),
                    resource.getRealPath(),
                    isHidden(resource.getRealPath(), upload, getAuthInfo(request)),
                    isRedirectUpload,
                    userAgent, remoteIp);
        } catch (PermanentHttpFailureException e) {
            if (e.getStatusCode().isSome(HttpStatus.SC_409_CONFLICT)) {
                String location = e.headers.get("X-Disk-Path");
                if (location != null) {
                    setLocationHeader(response, location);
                    response.setStatus(HttpStatus.SC_409_CONFLICT);
                }
            } else if (e.getStatusCode().isSome(HttpStatus.SC_423_LOCKED)) {
                Option<String> titleO = MpfsResponseParserUtils.getTitleFromMpfsResponse(
                        e.responseBody, UPLOAD_TRAFFIC_LIMIT_MPFS_CODES);
                titleO.ifPresent(title -> response.setHeader("Yandex-Disk-Limit", title));
            }
            throw e;
        }
    }

    private void setLocationHeader(WebdavResponse response, String location) {
        response.addHeader("Location", location);
    }

    private boolean isHidden(String realPath, Upload upload, AuthInfo authInfo) {
        return upload.hidden ||
                (authInfo.clientCapabilities.isUseAutohide() && invisiblePattern.matches(realPath));
    }

    private boolean isRedirectUpload(ClientCapabilities clientCapabilities, Upload upload) {
        if (upload.mode == Upload.Mode.DELTA) {
            // TODO: check that delta uploading doesn't support 302
            return false;
        }
        return clientCapabilities.isPutAlwaysRedirect() ||
                (clientCapabilities.isPutRedirect() &&
                        redirectPutThreshold.le(DataSize.fromBytes(upload.size.getOrElse(0L))));
    }

}
