package ru.yandex.chemodan.app.webdav.servlet;

import java.io.IOException;
import java.util.Enumeration;

import com.fasterxml.jackson.databind.JsonNode;
import lombok.AllArgsConstructor;
import lombok.SneakyThrows;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.auth.OurClient;
import ru.yandex.chemodan.app.webdav.filter.AuthenticationFilter;
import ru.yandex.chemodan.app.webdav.repository.MpfsResource;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
@AllArgsConstructor
public class UserinfoHandler implements DavMethodHandler {
    private static final Logger logger = LoggerFactory.getLogger(UserinfoHandler.class);

    private static final MapF<String, String> headers = Cf
            .map("X-Install-OS", "os")
            .plus1("X-Install-Manufacturer", "manufacturer")
            .plus1("X-Install-Product", "product")
            .plus1("X-Install-SerialNumber", "serial")
            .plus1("X-Install-Campaign", "campaign")
            .plus1("X-Install-Yms", "yms")
            .unmodifiable();

    private final MpfsClient mpfsClient;

    private final int uploadConcurrency;
    private final ListF<DenyRule> denyRules;

    @Override
    public void handle(WebdavRequest request, WebdavResponse response, MpfsResource resource)
            throws IOException, DavException
    {
        AuthInfo authInfo = AuthenticationFilter.getAuthInfo(request);
        //TODO: check trusted our client
        //TODO: add bonuses infp
        //TODO: maybe cache devices not to call mpfs on every request

        if (!authInfo.isAuthorized() && authInfo.authType != AuthInfo.AuthType.BANNED) {
            throw authInfo.buildUnauthorizedException();
        }

        Option<String> versionInfo = checkClientVersion(authInfo);

        maybeSetMobileInstalled(authInfo);
        updateDeviceInfo(request);

        response.setContentType("text/plain;charset=UTF-8");

        StringBuilder sb = new StringBuilder();

        authInfo.uid.forEach(v -> sb.append("uid:").append(v).append("\n"));
        authInfo.login.forEach(v -> sb.append("login:").append(v).append("\n"));

        sb.append("fio:")
                .append(authInfo.firstname.getOrElse(""))
                .append(" ")
                .append(authInfo.lastname.getOrElse(""))
                .append("\n");

        authInfo.firstname.forEach(v -> sb.append("firstname:").append(v).append("\n"));
        authInfo.lastname.forEach(v -> sb.append("lastname:").append(v).append("\n"));
        authInfo.country.forEach(v -> sb.append("country:").append(v).append("\n"));

        versionInfo.forEach(v -> sb.append("client_status:").append(v).append("\n"));

        sb.append("upload_concurrency:").append(uploadConcurrency).append("\n");

        sb.append("datasync_db_prefix:").append(authInfo.teamLogin.isPresent() ? "staff" : "").append("\n");
        sb.append("is_b2b:").append(authInfo.isB2b).append("\n");

        response.getWriter().print(sb.toString());
    }

    private void updateDeviceInfo(WebdavRequest request) {
        try {
            HttpServletRequestX reqX = HttpServletRequestX.wrap(request);

            AuthInfo authInfo = getAuthInfo(request);

            if (!authInfo.isOurClient()) {
                return;
            }

            Option<String> deviceId = reqX.getHeaderO("X-Install-DeviceID");
            OurClient.Type type = authInfo.ourClient.get().type;
            Option<String> installId = authInfo.ourClient.get().installId;

            if (deviceId.isPresent() && installId.isPresent() && type != OurClient.Type.UNKNOWN) {
                MapF<String, Object> params = Cf
                        .<String, Object>hashMap()
                        .plus1("install", installId.get())
                        .plus1("os_type", authInfo.ourClient.get().os);

                headers.forEach((header, param) -> reqX.getHeaderO(header).forEach(v -> params.put(param, v)));
                mpfsClient.userInstallDevice(authInfo, type.toString().toLowerCase(), deviceId.get(), params);
                updateOemInfo(authInfo, reqX.getHeaders("Yandex-OEM"));
            }
        } catch (Exception e) {
            logger.error("Failed to update device info: {}", e);
        }
    }

    private void updateOemInfo(AuthInfo authInfo, Enumeration<String> headers) {
        Cf.x(headers).forEachRemaining(header -> {
            try {
                JsonNode node = JsonNodeUtils.getNode(header);
                String serial = node.get("serial").asText();

                mpfsClient.userInstallDevice(
                        authInfo,
                        "storage",
                        serial,
                        Tuple2List.<String, Object>fromPairs(
                                "manufacturer", node.get("oem").asText(),
                                "serial", serial,
                                "size", node.get("size").asText(),
                                "check", node.get("check").asText()
                        ).toMap()
                );
            } catch (Exception e) {
                logger.error("Failed to update OEM info: {}", e);
            }
        });
    }

    private void maybeSetMobileInstalled(AuthInfo authInfo) {
        if (authInfo.ourClient.map(c -> c.type).isSome(OurClient.Type.MOBILE)) {
            try {
                mpfsClient.stateSet(authInfo, "mobile_installed", "1");
            } catch (PermanentHttpFailureException e) {
                if (e.responseBody.isPresent()
                        && e.responseBody.get().contains("user is not initialized"))
                {
                    return;
                }
                logger.error("Failed to set mobile_installed=1: {}", e);
            } catch (Exception e) {
                logger.error("Failed to set mobile_installed=1: {}", e);
            }
        }
    }

    private Option<String> checkClientVersion(AuthInfo authInfo) {
        if (!authInfo.isOurClient()) {
            return Option.empty();
        }

        if (authInfo.authType == AuthInfo.AuthType.BANNED) {
            return Option.of("banned");
        }

        OurClient ourClient = authInfo.ourClient.get();

        // only desktop has different statuses
        if(ourClient.type != OurClient.Type.DESKTOP) {
            return Option.empty();
        }

        for (DenyRule rule : denyRules) {
            Option<String> check = rule.check(ourClient);
            if (check.isPresent()) {
                return check;
            }
        }
        return Option.empty();
    }

    @Override
    public String method() {
        return "GET";
    }

    @Override
    public boolean matches(WebdavRequest request, MpfsResource resource) {
        return request.getParameter("userinfo") != null
                && (request.getPathInfo() == null
                    || request.getPathInfo().equals("")
                    || request.getPathInfo().equals("/")
                    // android mobile application makes such request
                    || request.getPathInfo().equals("//")
        );
    }

    @Override
    public int order() {
        return 1;
    }

    @Bendable
    public static class DenyRule {
        static final BenderJsonParser<DenyRule> P = Bender.jsonParser(DenyRule.class);

        @BenderPart
        private String os;
        @BenderPart
        private DenyResponse response;

        private int minVersion;
        private int maxVersion;

        public Option<String> check(OurClient ourClient) {
            return ourClient.build.filter(b -> b >= minVersion && b <= maxVersion && os.equals(ourClient.os)).flatMapO(b -> response.getStatusOrThrow());
        }

        @BenderPart
        public void setVersion(String version) {
            ListF<Integer> versions = Cf.list(version.split("-")).flatMap(Cf.Integer::parseSafe);
            minVersion = versions.first();
            maxVersion = versions.last();
        }
    }

    @AllArgsConstructor
    enum DenyResponse {
        FORBIDDEN(Either.left(HttpStatus.SC_403_FORBIDDEN)),
        NOT_AUTHORIZED(Either.left(HttpStatus.SC_401_UNAUTHORIZED)),
        DEPRECATED(Either.right("deprecated")),
        DISABLED(Either.right("disabled"));

        private final Either<Integer, String> response;

        public Option<String> getStatusOrThrow() {
            return response.left().map(DavException::new).fold(this::doThrow, Option::of);
        }

        @SneakyThrows
        private <T> T doThrow(Throwable e) {
            throw e;
        }
    }
}
