package ru.yandex.chemodan.app.webdav.servlet.index;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.mail.ContentType;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderDefaultValue;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.bender.parse.JacksonJsonNodeWrapper;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
public class MpfsIndexWriter {
    private static final Logger logger = LoggerFactory.getLogger(MpfsIndexWriter.class);

    private final JsonParser parser;
    private final ListF<String> extensions;
    private final String version;

    public MpfsIndexWriter(InputStream in, ListF<String> extensions, String version) throws IOException {
        this((new JsonFactory()).createParser(in), extensions, version);
    }

    public MpfsIndexWriter(JsonParser parser, ListF<String> extensions, String version) {
        this.extensions = extensions;
        this.version = version;
        this.parser = parser;
    }

    public void writeBinaryIndex(OutputStream out) {
        logger.debug("Write index with v={}, ext={}", version, extensions);

        try {
            JsonToken token = parser.nextToken();

            Check.equals(JsonToken.START_OBJECT, token, "root node of mpfs index must be object");

            int depth = 0;
            while (parser.hasCurrentToken()) {
                if (token.isStructStart()) depth++;
                if (token.isStructEnd()) depth--;
                if (depth == 1 && token.equals(JsonToken.FIELD_NAME) && parser.getCurrentName().equals("result")) {
                    parser.nextToken();

                    wireResultImpl(out);
                }
                token = parser.nextToken();
            }
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    //start at token = "["
    private void wireResultImpl(OutputStream out) throws IOException {
        JsonToken token = parser.getCurrentToken();

        Check.equals(JsonToken.START_ARRAY, token, "'result' node must be array");
        token = parser.nextToken();

        int depth = 1;
        while (depth > 0) {
            if (depth == 1 && token == JsonToken.START_OBJECT) {
                writeItemImpl(out);
            } else if (token.isStructStart()) {
                depth++;
            } else if (token.isStructEnd()) {
                depth--;
            }
            token = parser.nextToken();
        }
    }

    private void writeItemImpl(OutputStream out) throws IOException {
        JsonNode node = new ObjectMapper().readTree(parser);

        IndexItem item;
        if(node.get("type").asText().equals("dir")) {
            item = IndexItem.P.parseJson(new JacksonJsonNodeWrapper(node));
        } else {
            item = FileIndexItem.P.parseJson(new JacksonJsonNodeWrapper(node));
        }

        Check.isTrue(item.key.startsWith("/disk/"), "Key must start with /disk/: " + item.key);
        String path = StringUtils.removeStart(item.key, "/disk/");

        if (item.type == Type.DIR) {
            switch (item.op) {
                case DELETED: writeDeleted(out, path, item); break;
                case CHANGED: if (!"1".equals(version)) break; //TODO: check we need print dir changes
                case NEW: writeUpdateDir(out, path, item);
            }
        } else {
            switch (item.op) {
                case DELETED: writeDeleted(out, path, item); break;
                default: writeUpdateFile(out, path, (FileIndexItem) item);
            }
        }
    }

    private int addFlags(int type, IndexItem item) {
        if (type != 1 && type != 2) {
            return type;
        }
        int newType = type | getExtFlags(item);
        return newType;
    }

    private int getExtFlags(IndexItem item) {
        if (!"1".equals(version)) {
            // no flags in v2
            return 0;
        }
        int ext = 0;

        if (item.isPublic == 1) {
            ext |= 4;
        }
        if (item.isVisible == 0) {
            ext |= 8;
        }

        if (!"1".equals(version) && !"2".equals(version)) {
            // I don't think, it is really used
            if (item.broken.isSome(1)) {
                ext |= 16;
            }
        }

        if ("1".equals(version)) {
            if (item.broken.isSome(1)) {
                ext |= 128;
            }
            if (item.external_setprop.isSome(1)) {
                ext |= 256;
            }

            final int sharedFlag = 16;
            final int ownedFlag = 32;
            final int readOnlyFlag = 64;

            if (item.shared.isSome("group")) {
                ext |= sharedFlag;

                if (!item.rights.isSome(660)) {
                    ext |= readOnlyFlag;
                }
            } else if (item.shared.isSome("owner")) {
                ext |= sharedFlag;
                ext |= ownedFlag;
            }
        }

        if (item.type == Type.FILE && "1".equals(version) && extensions.containsTs("media")) {
            if (item instanceof FileIndexItem && ((FileIndexItem) item).hasPreview) {
                ext |= (512 + 1024);
            } else {
                ext |= 512;
            }
        }

        return ext;
    }

    private void writeUpdateFile(OutputStream out, String path, FileIndexItem item) throws IOException {
        writeLong(addFlags(1, item), out);

        if ("1".equals(version)) {
            writeStr(item.fid, out);
        }

        byte[] md5Bytes = Hex.decode(item.md5);
        if (md5Bytes.length != 16) {
            throw new BadIndexFormatException("Wrong size of md5. Must be 16. Size = " + md5Bytes.length);
        }
        out.write(md5Bytes);

        byte[] sha256Bytes = Hex.decode(item.sha256);
        if (sha256Bytes.length != 32) {
            throw new BadIndexFormatException("Wrong size of md5. Must be 16. Size = " + md5Bytes.length);
        }
        out.write(sha256Bytes);

        writeLong(item.size, out);
        writeStr(path, out);

        if ("1".equals(version) && extensions.containsTs("media")) {
            writeLong(item.mtime, out);
            writeLong(item.etime.getOrElse(0L), out);
            writeStr(item.mediaType.getOrElse(""), out);
            writeStr(item.mimetype.getOrElse(ContentType.APPLICATION_OCTET_STREAM.toString()), out);
        }
    }

    private void writeUpdateDir(OutputStream out, String path, IndexItem item) throws IOException {
        writeLong(addFlags(2, item), out);

        if ("1".equals(version)) {
            writeStr(item.fid, out);
        }

        writeStr(path, out);
    }

    private void writeDeleted(OutputStream out, String path, IndexItem item) throws IOException {
        writeLong(addFlags(0, item), out);

        writeStr(path, out);
    }

    private void writeStr(String string, OutputStream out) throws IOException {
        if(logger.isTraceEnabled()) {
            logger.trace("string: {}", string);
        }

        byte[] bytes = string.getBytes(CharsetUtils.UTF8_CHARSET);

        writeLong(bytes.length, out);

        if(logger.isTraceEnabled()) {
            logger.trace("bytes: [{}]", bytes.length);
        }
        out.write(bytes);
    }

    private void writeLong(long n, OutputStream out) throws IOException {
        if(logger.isTraceEnabled()) {
            logger.trace("number: {}", n);
        }

        if (n == 0) {
            out.write(0);
            return;
        }
        writeLong(n >> 7, n & 127, out);
    }

    private void writeLong(long high, long low, OutputStream out) throws IOException {
        if (high == 0) {
            out.write((int) low);
            return;
        }
        out.write((int) (low | 128));
        writeLong(high >> 7, high & 127, out);
    }

    @BenderBindAllFields
    private static class IndexItem {
        static final BenderJsonParser<IndexItem> P = Bender.jsonParser(IndexItem.class);

        String uid;
        String fid;
        @BenderPart(name = "visible")
        int isVisible;
        String key;
        Type type;
        @BenderPart(name = "public")
        @BenderDefaultValue("0")
        int isPublic;
        Operation op;

        Option<Integer> broken;
        @BenderPart(name = "external_setprop", strictName = true)
        Option<Integer> external_setprop;

        Option<Integer> rights;
        Option<String> shared;
    }

    @BenderBindAllFields
    private static class FileIndexItem extends IndexItem {
        static final BenderJsonParser<FileIndexItem> P = Bender.jsonParser(FileIndexItem.class);

        long size;
        String fid;
        String md5;
        String sha256;
        public long mtime;
        public Option<Long> etime;
        @BenderPart(name = "media_type", strictName = true)
        public Option<String> mediaType;
        public Option<String> mimetype;
        @BenderPart(name = "has_preview", strictName = true)
        public boolean hasPreview;
    }

    enum Type {
        FILE,
        DIR
    }

    enum Operation {
        NEW,
        CHANGED,
        DELETED
    }
}
