package ru.yandex.chemodan.app.webdav.servlet.index;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

import lombok.AllArgsConstructor;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.io.devnull.DevNullOutputStream;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
@RunWith(Parameterized.class)
public class MpfsIndexWriterTest {

    @AllArgsConstructor
    private static class TestData {

        final Callable<byte[]> indexBytes;
        final Callable<byte[]> v1Bytes;
        final Callable<byte[]> v2Bytes;
        final Callable<byte[]> v1MediaBtes;

    }

    @Parameterized.Parameters(name = "{index}")
    public static Iterable<Object[]> data() {
        byte[] bigFullIndex = download("https://proxy.sandbox.yandex-team.ru/208947192");

        return Cf.list(
                new TestData(
                        () -> bigFullIndex,
                        () -> download("https://proxy.sandbox.yandex-team.ru/208961889"),
                        () -> download("https://proxy.sandbox.yandex-team.ru/208965951"),
                        () -> download("https://proxy.sandbox.yandex-team.ru/208963832")
                ),
                classPathData("test/file_del"),
                classPathData("test/dir_del"),
                classPathData("test/mkdir")
        )
        .map(t -> new Object[]{t});
    }

    private static TestData classPathData(String base) {
        return new TestData(
                () -> resourceBytes(base + "/index.json"),
                () -> resourceBytes(base + "/v1.bin"),
                () -> resourceBytes(base + "/v2.bin"),
                () -> resourceBytes(base + "/v1_media.bin")
        );
    }

    private final TestData testData;

    public MpfsIndexWriterTest(TestData testData) {
        this.testData = testData;
    }

    @Test
    public void v1() throws Exception {
        test("1", Cf.list(), testData.indexBytes.call(), testData.v1Bytes.call());
    }

    @Test
    public void v1_media() throws Exception {
        test("1", Cf.list("media"), testData.indexBytes.call(), testData.v1MediaBtes.call());
    }

    @Test
    public void v2() throws Exception {
        test("2", Cf.list(), testData.indexBytes.call(), testData.v2Bytes.call());
    }

    @Test
    public void nullMimetype() throws IOException {
        DevNullOutputStream devNullOutputStream = new DevNullOutputStream();
        try(InputStream in = getClass().getResourceAsStream("test/null_mimetype_diff.json")) {
            new MpfsIndexWriter(in, Cf.list(), "2")
                    .writeBinaryIndex(devNullOutputStream);
        }
    }

    void test(String version, ListF<String> ext, byte[] jsonIndexBytes, byte[] rightBytes) throws IOException {
        MpfsIndexWriter writer = new MpfsIndexWriter(new ByteArrayInputStream(jsonIndexBytes), ext, version);

        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        writer.writeBinaryIndex(baos);


        String rightIndexStr = MpfsIndexReader.readIndex(version, ext, new ByteArrayInputStream(rightBytes));
        String myIndexStr = MpfsIndexReader.readIndex(version, ext, new ByteArrayInputStream(baos.toByteArray()));

//        new File2("/Users/tolmalev/arcadia/disk/webdav/src/main/java/ru/yandex/chemodan/app/webdav/servlet/index/right" + version + "_" + ext.mkString("_") + ".txt").write(rightIndexStr);
//        new File2("/Users/tolmalev/arcadia/disk/webdav/src/main/java/ru/yandex/chemodan/app/webdav/servlet/index/my" + version + "_" + ext.mkString("_") + ".txt").write(myIndexStr);

//        Assert.equals(rightIndexStr, myIndexStr);

        Assert.assertArrayEquals(
                rightBytes,
                baos.toByteArray()
        );
    }

    private static byte[] download(String url) {
        return ApacheHttpClientUtils.download(url, Timeout.timeout(10, TimeUnit.SECONDS), true);
    }

    private static byte[] resourceBytes(String name) {
        return new ClassPathResourceInputStreamSource(MpfsIndexWriterTest.class, name).readBytes();
    }
}
