package ru.yandex.chemodan.app.worker2;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.OnetimeTaskUtils;
import ru.yandex.chemodan.core.worker.python.onetime.ConfigurableOnetimeTaskRegistry;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.JobInfoValue;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.commune.bazinga.scheduler.TaskCategory;
import ru.yandex.commune.json.JsonNumber;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author vavinov
 */
public class OnetimeTaskServlet extends HttpServlet {
    private static final Logger logger = LoggerFactory.getLogger(OnetimeTaskServlet.class);

    @Autowired
    private ConfigurableOnetimeTaskRegistry configurableOnetimeTaskRegistry;

    private final BazingaTaskManager bazingaTaskManager;
    private final BazingaStorage bazingaStorage;

    public OnetimeTaskServlet(BazingaTaskManager bazingaTaskManager, BazingaStorage bazingaStorage) {
        this.bazingaTaskManager = bazingaTaskManager;
        this.bazingaStorage = bazingaStorage;
    }

    private static MapF<String, String> prefixedMap(String prefix, Tuple2List<String, String> list) {
        return list
                .filterBy1(Cf.String.startsWithF(prefix))
                .map1(StringUtils.substringF(prefix.length()))
                .toMap();
    }

    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        HttpServletRequestX reqX = HttpServletRequestX.wrap(req);

        final OnetimeTask task;

        if (reqX.getParameterO("retry").isPresent()) {
            FullJobId fullJobId = FullJobId.parse(reqX.getParameter("retry"));

            task = OnetimeTaskUtils.makeOnetimeTask(fullJobId.getTaskId(),
                    bazingaStorage.findOnetimeJob(fullJobId).get().getParameters());

        } else if (configurableOnetimeTaskRegistry.getTask(reqX.getParameter("task")).isPresent()) {
            task = configurableOnetimeTaskRegistry.getTask(reqX.getParameter("task")).get()
                    .makeCopy(Cf.list(reqX.getParameterValues("arg.param")));
        } else {
            task = OnetimeTaskUtils.makeOnetimeTask(new TaskId(reqX.getParameter("task")),
                    new JsonObject(prefixedMap("arg.", reqX.getParameterFlatMap())
                            .mapValues((Function<String, JsonValue>) s -> {
                                if (StringUtils.isNumericArabic(s)) {
                                    return JsonNumber.valueOf(Long.parseLong(s));
                                } else {
                                    return JsonString.valueOf(s);
                                }
                            }).entries()).serialize());
        }

        TaskCategory category = reqX.getParameterO("category")
                .map(TaskCategory.consF())
                .getOrElse(task.getTaskCategory());

        Instant scheduleTime = reqX.getParameterO("when")
                .map(TimeUtils.instant.parseF())
                .getOrElse(Instant.now());

        int priority = reqX.getParameterO("priority").map(Cf.Integer.parseF()).getOrElse(task.priority());

        Option<String> group = reqX.getParameterO("group");
        FullJobId id = bazingaTaskManager.schedule(task, category, scheduleTime, priority, true, group);

        resp.setHeader("Location", UrlUtils.addParameter(reqX.getTrueRequestUrl(), "id", id.toSerializedString()));
        resp.setStatus(HttpStatus.SC_202_ACCEPTED);

        resp.getWriter().println(id.toSerializedString());
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        if (!req.getPathInfo().startsWith("/")) {
            resp.setStatus(HttpStatus.SC_400_BAD_REQUEST);
            return;
        }

        HttpServletRequestX reqX = HttpServletRequestX.wrap(req);

        ListF<String> path = Cf.x(req.getPathInfo().substring(1).split("/"));

        if (path.first().equals("id")) {
            String id = req.getPathInfo().substring("/id/".length()); // XXX
            Option<OnetimeJob> job = bazingaTaskManager.getOnetimeJob(FullJobId.parse(id));
            if (job.isPresent()) {
                resp.getWriter().println(job.get());
            } else {
                resp.setStatus(HttpStatus.SC_404_NOT_FOUND);
            }

        } else if (path.first().equals("group")) {
            String group = path.get(1);
            if (path.size() > 2 && path.get(2).equals("stat")) {
                // XXX inefficient
                resp.getWriter().println(bazingaTaskManager
                        .getJobsByGroup(group, SqlLimits.all())
                        .groupBy(OnetimeJob.getValueF().andThen(JobInfoValue.getStatusF()))
                        .mapValues(Cf.List.sizeF())
                        .entries()
                        .sortedBy1()
                        .mkString("\n", ": "));

            } else {
                for (OnetimeJob j : bazingaTaskManager.getJobsByGroup(group, SqlLimits.first(100))) {
                    resp.getWriter().println(j.getId() + " " + j.getValue().getStatus()
                            + " attempt=" + j.getAttempt().getOrElse(0)
                            + " " + j.getParameters());
                }
            }

        } else if (path.first().equals("list")) {
            TaskId taskId = new TaskId(reqX.getParameter("task"));
            Option<JobStatus> status = reqX.getParameterO("status").map(JobStatus.R.valueOfF());
            Option<Integer> limitO = reqX.getIntParameterO("limit");

            ListF<OnetimeJob> jobs = bazingaStorage.findLatestOnetimeJobs(taskId, status,
                    limitO.isPresent() ? SqlLimits.first(limitO.get()) : SqlLimits.all());

            for (OnetimeJob job : jobs) {
                resp.getWriter().println(job.getId()
                        + "\t" + job.getValue().getStatus().name().toLowerCase()
                        + "\t" + job.getParameters());
            }

        } else {
            resp.setStatus(HttpStatus.SC_400_BAD_REQUEST);
        }
    }
}
