package ru.yandex.chemodan.app.worker2.queued;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.mongo.bender.MongoId;
import ru.yandex.inside.elliptics.EllipticsFilename;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

/**
 * @author tolmalev
 */
@BenderBindAllFields
public class LongTask {
    @MongoId
    public final String id;

    public final TaskId taskId;

    public final Option<String> description;

    public final String ellipticsNamespace;
    public final String ellipticsKey;
    public final String ellipticsFilename;

    public final State state;

    public final int scheduledThreshold;

    public final long totalTasksCount;
    public final long scheduledTasksCount;
    public final long remainingTasksCount;

    public final long ellipticsOffset;

    public final int maxLineLengthBytes;

    public final Option<String> errorHostname;

    public final Option<String> errorMessage;
    public final Option<String> errorStackTrace;

    public final Option<Integer> failsCount;

    public final boolean initialized;

    private LongTask(String id, TaskId taskId, Option<String> description,
            String ellipticsNamespace, String ellipticsKey,
            String ellipticsFilename, State state, int scheduledThreshold, long totalTasksCount,
            long scheduledTasksCount, long remainingTasksCount, long ellipticsOffset,
            int maxLineLengthBytes, Option<String> errorHostname, Option<String> errorMessage,
            Option<String> errorStackTrace,
            Option<Integer> failsCount, boolean initialized)
    {
        this.id = id;
        this.taskId = taskId;
        this.description = description;
        this.ellipticsNamespace = ellipticsNamespace;
        this.ellipticsKey = ellipticsKey;
        this.ellipticsFilename = ellipticsFilename;
        this.state = state;
        this.scheduledThreshold = scheduledThreshold;
        this.totalTasksCount = totalTasksCount;
        this.scheduledTasksCount = scheduledTasksCount;
        this.remainingTasksCount = remainingTasksCount;
        this.ellipticsOffset = ellipticsOffset;
        this.maxLineLengthBytes = maxLineLengthBytes;
        this.errorHostname = errorHostname;
        this.errorMessage = errorMessage;
        this.errorStackTrace = errorStackTrace;
        this.failsCount = failsCount;
        this.initialized = initialized;
    }

    public LongTask(String id, TaskId taskId, Option<String> description, String ellipticsNamespace,
            String ellipticsKey,
            String ellipticsFilename, int scheduledThreshold)
    {
        this.id = id;
        this.taskId = taskId;
        this.description = description;
        this.ellipticsNamespace = ellipticsNamespace;
        this.ellipticsKey = ellipticsKey;
        this.ellipticsFilename = ellipticsFilename;
        this.scheduledThreshold = scheduledThreshold;

        this.state = State.IN_PROGRESS;
        this.initialized = false;
        this.maxLineLengthBytes = 0;
        this.totalTasksCount = 0;
        this.scheduledTasksCount = 0;
        this.remainingTasksCount = totalTasksCount;
        this.ellipticsOffset = 0;
        this.errorHostname = Option.empty();
        this.errorMessage = Option.empty();
        this.errorStackTrace = Option.empty();
        this.failsCount = Option.empty();
    }

    public LongTask initialized(int totalTasksCount, int maxLineLengthBytes) {
        return new LongTask(id, taskId, description, ellipticsNamespace, ellipticsKey, ellipticsFilename,
                state, scheduledThreshold,
                totalTasksCount, 0, totalTasksCount,
                ellipticsOffset, maxLineLengthBytes, errorHostname, errorMessage,
                errorStackTrace, failsCount, true);
    }

    public LongTask withIncreasedOffset(long processedBytes) {
        return new LongTask(id, taskId, description, ellipticsNamespace, ellipticsKey, ellipticsFilename, state,
                scheduledThreshold, totalTasksCount, scheduledTasksCount, remainingTasksCount,
                ellipticsOffset + processedBytes, maxLineLengthBytes, errorHostname, errorMessage, errorStackTrace,
                failsCount, initialized);
    }

    public LongTask withScheduledThreshold(int newScheduledThreshold) {
        return new LongTask(id, taskId, description, ellipticsNamespace, ellipticsKey, ellipticsFilename, state,
                newScheduledThreshold, totalTasksCount, scheduledTasksCount, remainingTasksCount,
                ellipticsOffset, maxLineLengthBytes, errorHostname, errorMessage, errorStackTrace,
                failsCount, initialized);
    }

    public LongTask withState(State newState) {
        return new LongTask(id, taskId, description, ellipticsNamespace, ellipticsKey, ellipticsFilename,
                newState, scheduledThreshold, totalTasksCount, scheduledTasksCount,
                remainingTasksCount, ellipticsOffset, maxLineLengthBytes, errorHostname, errorMessage,
                errorStackTrace, failsCount, initialized);
    }

    public LongTask withScheduledJobs(int processedJobsCount) {
        return new LongTask(id, taskId, description, ellipticsNamespace, ellipticsKey, ellipticsFilename, state,
                scheduledThreshold, totalTasksCount,
                scheduledTasksCount + processedJobsCount,
                remainingTasksCount - processedJobsCount,
                ellipticsOffset, maxLineLengthBytes,
                errorHostname, errorMessage, errorStackTrace, failsCount,
                initialized);
    }

    public LongTask withNoError() {
        return new LongTask(id, taskId, description, ellipticsNamespace, ellipticsKey, ellipticsFilename,
                state,
                scheduledThreshold, totalTasksCount, scheduledTasksCount, remainingTasksCount,
                ellipticsOffset, maxLineLengthBytes,
                Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                initialized);
    }

    public LongTask withError(RuntimeException e, String hostname) {
        return new LongTask(id, taskId, description, ellipticsNamespace, ellipticsKey, ellipticsFilename,
                State.ERROR,
                scheduledThreshold, totalTasksCount, scheduledTasksCount, remainingTasksCount,
                ellipticsOffset, maxLineLengthBytes,
                Option.of(hostname), Option.of(e.getMessage()), Option.of(ExceptionUtils.getStackTrace(e)),
                failsCount, initialized);
    }

    public LongTask withIncFailesCount(RuntimeException e, String hostname) {
        return new LongTask(id, taskId, description, ellipticsNamespace, ellipticsKey, ellipticsFilename,
                state,
                scheduledThreshold, totalTasksCount, scheduledTasksCount, remainingTasksCount,
                ellipticsOffset, maxLineLengthBytes,
                Option.of(hostname), Option.of(e.getMessage()), Option.of(ExceptionUtils.getStackTrace(e)),
                Option.of(failsCount.getOrElse(0) + 1), initialized);
    }

    public EllipticsFilename getEllipticsFilename() {
        return new EllipticsFilename(ellipticsNamespace, ellipticsFilename, ellipticsKey);
    }

    public static Function<LongTask, State> getStateF() {
        return longTask -> longTask.state;
    }

    public static Function<LongTask, TaskId> getTaskIdF() {
        return longTask -> longTask.taskId;
    }

    public enum State {
        IN_PROGRESS,
        FINISHED,
        ERROR,
        SUSPENDED
    }
}
