package ru.yandex.chemodan.app.worker2.queued;

import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.ReadPreference;
import com.mongodb.ServerAddress;
import com.mongodb.WriteConcern;
import org.apache.http.client.HttpClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.worker2.ProcessLongTasks;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.core.worker.python.onetime.ConfigurableOnetimeTaskRegistry;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.commune.admin.web.AdminApp;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.mongo.MongoMediaDevelopmentUtils;
import ru.yandex.commune.mongo.MongoUtils;
import ru.yandex.inside.admin.conductor.Conductor;
import ru.yandex.inside.elliptics.EllipticsProxyClient;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.lang.tsb.YandexToStringBuilder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;

/**
 * @author tolmalev
 */
@Configuration
public class LongTasksContextConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(LongTasksContextConfiguration.class);

    @Autowired
    private AdminApp adminApp;

    @Bean
    public LongTasksDao longTasksDao(
            @Qualifier("longTasksMongoClient")
            MongoClient longTasksMongoClient,
            @Value("${long-tasks.mongo.db}")
            String databaseName)
    {
        return new LongTasksDao(longTasksMongoClient.getDB(databaseName));
    }

    @Bean
    public LongTasksAdminPage longTasksAdminPage(LongTasksManager longTasksManager) {
        LongTasksAdminPage page = new LongTasksAdminPage(longTasksManager);
        adminApp.registerAdminPage(page);
        return page;
    }

    @Bean
    public ProcessLongTasks processLongTasks(LongTasksManager longTasksManager) {
        return new ProcessLongTasks(longTasksManager);
    }

    @Bean
    public LongTasksManager longTasksManager(
            LongTasksDao longTasksDao,
            BazingaTaskManager bazingaTaskManager,
            BazingaStorage bazingaStorage,
            @Qualifier("workerEllipticsClient")
            EllipticsProxyClient ellipticsProxyClient,
            @Value("${elliptics.namespace}")
            String ellipticsNamespace,
            ConfigurableOnetimeTaskRegistry configurableOnetimeTaskRegistry)
    {
        return new LongTasksManager(longTasksDao, bazingaTaskManager, bazingaStorage,
                ellipticsProxyClient, ellipticsNamespace, configurableOnetimeTaskRegistry);
    }

    @Bean
    public EllipticsProxyClient workerEllipticsClient(
            @Value("${elliptics.url}")
            String ellipticsUrl,
            @Value("${elliptics.port.read}")
            int ellipticsReadPort,
            @Value("${elliptics.port.write}")
            int ellipticsWritePort,
            @Value("${elliptics.authorization.header}")
            String authHeader)
    {
        HttpClient httpClient = ellipticsHttpClientConfigurator().configure();

        EllipticsProxyClient ellipticsProxyClient = new EllipticsProxyClient(ellipticsUrl, httpClient,
                ellipticsReadPort, ellipticsWritePort, new InstrumentMap(), true);
        ellipticsProxyClient.setAuthorizationHeader(authHeader);
        return ellipticsProxyClient;
    }

    @Bean
    @OverridableValuePrefix("elliptics")
    public HttpClientConfigurator ellipticsHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    public MongoClient longTasksMongoClient(
            @Value("${mongo.hosts}")
                    String hosts,
            @Value("${mongo.port}")
                    IpPort port,
            @Value("${mongo.connections.per.host:-50}")
                    int connectionsPerHost,
            @Value("${mongo.connect.timeout.ms:-5000}")
                    int connectTimeoutMs,
            @Value("${mongo.socket.timeout.ms:-5000}")
                    int socketTimeoutMs,
            @Value("${mongo.pool.timeout.ms:-5000}")
                    int poolWaitTimeoutMs,
            @Value("${mongo.threads.block.multiplier:-20}")
                    int threadsAllowedToBlockForConnectionMultiplier,
            @Value("${mongo.w:-2}")
                    int w,
            @Value("${mongo.fsync:-false}")
                    boolean fsync,
            @Value("${mongo.wtimeout.ms:-2500}")
                    int wtimeoutMs,
            Conductor conductor,
            EnvironmentType environmentType)
    {
        ListF<ServerAddress> hostList;
        if (!hosts.isEmpty()) {
            hostList = conductor.resolveHostsFromString(hosts).map(MongoUtils.consServerAddressF(port));
        } else if (environmentType == EnvironmentType.DEVELOPMENT) {
            hostList = MongoMediaDevelopmentUtils.hosts().map(MongoUtils.consServerAddressF(port));
        } else {
            throw new RuntimeException("No mongo hosts defined");
        }

        MongoClientOptions mongoOptions = MongoClientOptions.builder()
                .connectionsPerHost(connectionsPerHost)
                .connectTimeout(connectTimeoutMs)
                .socketTimeout(socketTimeoutMs)
                .maxWaitTime(poolWaitTimeoutMs)
                .threadsAllowedToBlockForConnectionMultiplier(threadsAllowedToBlockForConnectionMultiplier)
                .writeConcern(new WriteConcern(w, wtimeoutMs, fsync))
                .readPreference(ReadPreference.primaryPreferred())
                .build();

        logger.info("Starting mongo client to hosts {} with options: {}", hostList,
                YandexToStringBuilder.reflectionToStringValueObject(mongoOptions));

        return new MongoClient(hostList, mongoOptions);
    }
}
