package ru.yandex.chemodan.app.worker2.queued;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function1I;
import ru.yandex.chemodan.OnetimeTaskUtils;
import ru.yandex.chemodan.core.worker.python.onetime.ConfigurableOnetimeTask;
import ru.yandex.chemodan.core.worker.python.onetime.ConfigurableOnetimeTaskRegistry;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.BazingaUtils;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.inside.elliptics.EllipticsFilename;
import ru.yandex.inside.elliptics.EllipticsProxyClient;
import ru.yandex.misc.bender.parse.BenderParser;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.IoFunction;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.HostnameUtils;

/**
 * @author tolmalev
 */
public class LongTasksManager {
    private static final Logger logger = LoggerFactory.getLogger(LongTasksManager.class);

    private static final BenderParser<ConfigurableOnetimeTask.Parameters> CONFIGURABLE_ONETIME_TASKS_PARAMETERS_PARSER =
            BazingaBender.mapper.createParser(ConfigurableOnetimeTask.Parameters.class);

    private final LongTasksDao longTasksDao;
    private final BazingaTaskManager bazingaTaskManager;
    private final BazingaStorage bazingaStorage;
    private final EllipticsProxyClient ellipticsProxyClient;
    private final String ellipticsNamespace;
    private final ConfigurableOnetimeTaskRegistry configurableOnetimeTaskRegistry;

    public LongTasksManager(LongTasksDao longTasksDao, BazingaTaskManager bazingaTaskManager,
            BazingaStorage bazingaStorage, EllipticsProxyClient ellipticsProxyClient,
            String ellipticsNamespace,
            ConfigurableOnetimeTaskRegistry configurableOnetimeTaskRegistry)
    {
        this.longTasksDao = longTasksDao;
        this.bazingaTaskManager = bazingaTaskManager;
        this.bazingaStorage = bazingaStorage;
        this.ellipticsProxyClient = ellipticsProxyClient;
        this.ellipticsNamespace = ellipticsNamespace;
        this.configurableOnetimeTaskRegistry = configurableOnetimeTaskRegistry;
    }

    public LongTask createLongTask(TaskId taskId, String filePath, int scheduledThreshold,
            Option<String> description)
    {
        String longTaskId = taskId + "_" + Instant.now();

        File2 parametersFile = new File2(filePath);

        logger.info("Uploading file to elliptics");
        EllipticsFilename filename =
                new EllipticsFilename(ellipticsNamespace, longTaskId + "-parameters");

        String ellipticsKey = ellipticsProxyClient.upload(filename, parametersFile).getKey();

        LongTask task = new LongTask(
                longTaskId,
                taskId,
                description,
                filename.getNamespace().get(), ellipticsKey, filename.getFilename(),
                scheduledThreshold);
        longTasksDao.insert(task);
        return task;
    }

    public void suspendLongTask(String taskId) {
        longTasksDao.updateTask(
                longTasksDao.find(taskId).get().withState(LongTask.State.SUSPENDED));
    }

    public void resumeLongTask(String taskId) {
        longTasksDao.updateTask(longTasksDao.find(taskId).get()
                .withNoError()
                .withState(LongTask.State.IN_PROGRESS));
    }

    public void removeTask(String longTaskId) {
        LongTask task = longTasksDao.find(longTaskId).get();
        logger.info("Removing file with parameters from elliptics");
        ellipticsProxyClient.delete(task.getEllipticsFilename());
        logger.info("Remove task from mongo");
        longTasksDao.remove(longTaskId);
    }

    public void changeScheduledThreshold(String longTaskId, int newScheduledThreshold) {
        Option<LongTask> task0 = longTasksDao.find(longTaskId);
        if (task0.isPresent()) {
            longTasksDao.updateTask(task0.get().withScheduledThreshold(newScheduledThreshold));
        }
    }

    public void processLongTasks() {
        logger.info("Calculating stats of all tasks");

        ListF<LongTask> longTasks = longTasksDao.list();

        SetF<TaskId> taskIds = longTasks
                .filter(LongTask.getStateF().andThenEquals(LongTask.State.IN_PROGRESS))
                .map(LongTask.getTaskIdF())
                .unique();

        logger.info("Calculating queue lengths");
        MapF<TaskId, Integer> queues = taskIds.toMapMappingToValue(((Function1I<TaskId>) taskId -> {
            // fixes problem for removed 'Task' suffix
            return bazingaStorage.findNotCompletedOnetimeJobCount(
                    new TaskId(BazingaUtils.simpleClassNameToTaskId(taskId.getId())));
        }).asFunction());

        logger.info("Current queues: {}", queues);

        for (LongTask task : longTasks) {
            if (task.state == LongTask.State.IN_PROGRESS) {
                try {
                    logger.info("Processing task {}", task.id);

                    if (!task.initialized) {
                        task = initializeTask(task);
                    }

                    int queue = queues.getTs(task.taskId);

                    logger.info("Current queue size for task {} is {}. Threshold: {}", task.id,
                            queue, task.scheduledThreshold);

                    if (queue < task.scheduledThreshold) {
                        logger.info("Scheduling new chunk of jobs");
                        longTasksDao.updateTask(scheduleJobs(task).withNoError());
                    } else {
                        logger.info("Queue is full. Not scheduling new jobs");
                    }
                } catch (RuntimeException e) {
                    logger.error("Failed to process task: {}", e);
                    if (task.failsCount.getOrElse(0) > 5) {
                        longTasksDao.updateTask(task.withError(e, HostnameUtils.localHostname()));
                    } else {
                        longTasksDao.updateTask(task.withIncFailesCount(e, HostnameUtils.localHostname()));
                    }
                }
            }
        }
    }

    private LongTask initializeTask(final LongTask task) {
        logger.info("Initializing task {}", task.id);

        LongTask newTask = File2.withNewTempFile((IoFunction<File2, LongTask>) file2 -> {
            logger.info("Downloading full file from elliptics");
            ellipticsProxyClient
                    .download(task.getEllipticsFilename(), file2.asOutputStreamTool());

            logger.info("Reading file to get lines count and max line length");
            BufferedReader br = new BufferedReader(new InputStreamReader(file2.getInput()));

            int linesCount = 0;
            int maxLineLength = 0;

            String line;
            while ((line = br.readLine()) != null) {
                linesCount++;
                maxLineLength = Math.max(maxLineLength, line.length());
            }

            br.close();

            logger.info("Jobs count: {}, max line length: {}", linesCount, maxLineLength);

            return task.initialized(linesCount, maxLineLength);
        });
        longTasksDao.updateTask(newTask);
        return newTask;
    }

    public LongTask scheduleJobs(final LongTask task) {
        int jobsCount = (int) Math.min(task.scheduledThreshold, task.remainingTasksCount);
        if (jobsCount == 0) {
            return task.withState(LongTask.State.FINISHED);
        }
        logger.info("Scheduling {} jobs for task {}", jobsCount, task.taskId);
        // some more than necessary
        long downloadSize = (long) jobsCount * (task.maxLineLengthBytes + 5);

        logger.info("Downloading {} bytes from elliptic with offset {}",
                DataSize.fromBytes(downloadSize).toPrettyString(), task.ellipticsOffset);

        long processedBytes = File2.withNewTempFile(file2 -> {
            logger.info("Scheduling jobs");
            ellipticsProxyClient.download(task.getEllipticsFilename(),
                    file2.asOutputStreamTool(), 0,
                    task.ellipticsOffset, Option.of(downloadSize));

            try {
                RandomAccessFile raf = new RandomAccessFile(file2.getFile(), "r");
                for (int i = 0; i < jobsCount; i++) {
                    String parameters = raf.readLine();
                    if (!parameters.contains("{")) {
                        // Parameters is uid
                        parameters = "{\"uid\": " + parameters + "}";
                    }

                    OnetimeTask onetimeTask;
                    if (configurableOnetimeTaskRegistry.getTask(task.taskId.getId()).isPresent()) {
                        onetimeTask = configurableOnetimeTaskRegistry.getTask(task.taskId.getId()).get()
                                .makeCopy(CONFIGURABLE_ONETIME_TASKS_PARAMETERS_PARSER.parseJson(parameters));
                    } else {
                        onetimeTask = OnetimeTaskUtils.makeOnetimeTask(task.taskId, parameters);
                    }
                    bazingaTaskManager.schedule(onetimeTask);
                }
                return raf.getFilePointer();
            } catch (IOException e) {
                throw IoUtils.translate(e);
            }
        });

        logger.info("All jobs scheduled. Processed {} bytes", processedBytes);

        LongTask newTask = task
                .withIncreasedOffset(processedBytes)
                .withScheduledJobs(jobsCount);

        if (newTask.remainingTasksCount == 0) {
            logger.info("Task finished");
            newTask = newTask.withState(LongTask.State.FINISHED);
        }

        return newTask;
    }

    public ListF<LongTask> listTasks() {
        return longTasksDao.list();
    }
}
