package ru.yandex.chemodan.app.worker2.stat;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.chrono.ISOChronology;
import org.springframework.jdbc.core.ColumnMapRowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.commune.bazinga.BazingaWorkerApp;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.impl.worker.WorkerTaskRegistry;
import ru.yandex.commune.bazinga.scheduler.CronTask;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleCron;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.primitives.registry.ZkRegistry;
import ru.yandex.inside.statface.StatfaceClient;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.spring.jdbc.RowMapperResultSetExtractor2;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author tolmalev
 */
public class YqlStatTasksManager extends ZkRegistry<String, YqlStatTaskInfo> {
    private static final Logger logger = LoggerFactory.getLogger(YqlStatTasksManager.class);

    private static final String PREFIX = "statface_";

    private final BazingaWorkerApp bazingaWorkerApp;
    private final StatfaceClient statfaceClient;
    private final ExecutorService executors;
    private final String yqlDatasourceUrl;
    private final String yqlToken;
    private final String recalcYqlToken;

    public YqlStatTasksManager(
            ZkPath zkPath,
            BazingaWorkerApp bazingaWorkerApp,
            StatfaceClient statfaceClient,
            String yqlDatasourceUrl,
            String yqlToken,
            String recalcYqlToken)
    {
        super(zkPath, BazingaBender.mapper.createParserSerializer(YqlStatTaskInfo.class), task -> task.id, s -> s);

        this.bazingaWorkerApp = bazingaWorkerApp;
        this.statfaceClient = statfaceClient;
        this.yqlToken = yqlToken;
        this.yqlDatasourceUrl = yqlDatasourceUrl;
        this.recalcYqlToken = recalcYqlToken;
        this.executors = Executors.newFixedThreadPool(10);

        super.addListener(s -> updateTasks());
    }

    public void updateTasks() {
        ListF<CronTask> dynTasks = getAll().map(this::consTask);

        WorkerTaskRegistry taskRegistry = bazingaWorkerApp.getWorkerTaskRegistry();
        ListF<CronTask> existingTasks = taskRegistry
                .getCronTasks()
                .filter(task -> task.id().toString().startsWith(PREFIX));

        taskRegistry.addTasks(dynTasks.toMapMappingToKey(CronTask::id), Cf.map(), Cf.map());

        ListF<TaskId> toRemove = existingTasks.map(CronTask::id).unique().minus(dynTasks.map(CronTask::id)).toList();

        bazingaWorkerApp.getBazingaWorker().suspendTaskScheduling(toRemove);
        bazingaWorkerApp.getBazingaWorker().resumeTaskScheduling(dynTasks, Cf.list());
    }

    public void executeYqlStatForSomeDays(String id, Instant startDate, Instant endDate) {
        Check.isTrue(startDate.isBefore(endDate) || startDate.isEqual(endDate), "Start date must be less than end");

        createOrUpdateConfig(id);

        ListF<CompletableFuture> futures = Cf.arrayList();

        String requestId = RequestIdStack.current().getOrElse(() -> Random2.R.nextAlnum(10));

        Instant day = endDate;
        while (true) {
            String dayStr = formatDateForYql(day);

            futures.add(CompletableFuture.runAsync(() -> {
                RequestIdStack.withRequestId(requestId,
                        () -> executeYqlStat(id, dayStr, dayStr, StatfaceClient.Scale.DAY, true));
            }, executors));

            if (isMonday(day)) {
                Instant weekAgoDay = day.minus(Duration.standardDays(7));
                String weekAgoDayStr = formatDateForYql(weekAgoDay);

                Instant sunday = day.minus(Duration.standardDays(1));
                String sundayStr = formatDateForYql(sunday);

                futures.add(CompletableFuture.runAsync(() -> {
                    RequestIdStack.withRequestId(requestId,
                            () -> executeYqlStat(id, weekAgoDayStr, sundayStr, StatfaceClient.Scale.WEEK, true));
                }, executors));
            }

            day = day.minus(Duration.standardDays(1));

            if (dayStr.equals(formatDateForYql(startDate))) {
                break;
            }
        }

        ListF<String> errors = Cf.arrayList();

        futures.forEach(f -> {
            try {
                f.get();
            } catch (Exception e) {
                errors.add(ExceptionUtils.getAllMessages(e));
            }
        });

        if (errors.isNotEmpty()) {
            logger.error("Failed for some days");
            errors.forEach(logger::error);
        }
    }

    private void createOrUpdateConfig(String id) {
        YqlStatTaskInfo taskInfo = getO(id).getOrThrow("Unknown yql stat id ", id);

        statfaceClient.createOrUpdateReport(taskInfo.reportName,
                StatfaceClient.Scale.DAY, taskInfo.getReportConfig());

        statfaceClient.createOrUpdateReport(taskInfo.reportName,
                StatfaceClient.Scale.WEEK, taskInfo.getReportConfig());
    }

    public void executeYqlStat(String id) {
        Instant now = Instant.now();
        Instant prevDay = now.minus(Duration.standardDays(1));

        String prevDateStr = formatDateForYql(prevDay);
        createOrUpdateConfig(id);
        executeYqlStat(id, prevDateStr, prevDateStr, StatfaceClient.Scale.DAY, false);

        if (isMonday(now)) {
            Instant weekAgoDay = now.minus(Duration.standardDays(7));
            String weekAgoDayStr = formatDateForYql(weekAgoDay);

            Instant sunday = now.minus(Duration.standardDays(1));
            String sundayStr = formatDateForYql(sunday);

            executeYqlStat(id, weekAgoDayStr, sundayStr, StatfaceClient.Scale.WEEK, false);
        }
    }

    static boolean isMonday(Instant day) {
        return getChronology().dayOfWeek().get(day.getMillis()) == 1;
    }

    static ISOChronology getChronology() {
        return ISOChronology.getInstance(TimeUtils.EUROPE_MOSCOW_TIME_ZONE);
    }

    public static String formatDateForYql(Instant date) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        simpleDateFormat.setTimeZone(TimeZone.getTimeZone("Europe/Moscow"));

        return simpleDateFormat.format(date.toDate());
    }

    public static Instant parseDateFromYql(String date) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        simpleDateFormat.setTimeZone(TimeZone.getTimeZone("Europe/Moscow"));

        try {
            return new Instant(simpleDateFormat.parse(date)).plus(Duration.standardHours(2));
        } catch (ParseException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private void executeYqlStat(String id, String dateFrom, String dateTo, StatfaceClient.Scale scale,
            boolean isRecalc)
    {
        YqlStatTaskInfo taskInfo = getO(id).getOrThrow("Unknown yql stat id ", id);
        logger.debug("Going to execute yql stat task {} for {}-{}", id, dateFrom, dateTo);

        try {
            Class.forName("ru.yandex.yql.YqlDriver");
        } catch (ClassNotFoundException e) {
            throw ExceptionUtils.translate(e);
        }
        String dateForLog = dateFrom;
        if (!dateFrom.equals(dateTo)) {
            dateForLog += "-" + dateTo;
        }

        String token = isRecalc ? recalcYqlToken : yqlToken;

        try (Connection conn = DriverManager.getConnection(yqlDatasourceUrl, "unused", token);
             PreparedStatement stmt = conn.prepareStatement(prepareYql(taskInfo.yql, dateFrom, dateTo)))
        {
            Instant start = Instant.now();

            logger.debug("Executing YQL for date={}", dateForLog);
            ResultSet rs = stmt.executeQuery();
            ListF<Map<String, Object>> data =
                    new RowMapperResultSetExtractor2<>(new ColumnMapRowMapper()).extractData(rs);

            ListF<Map<String, Object>> dataWithFielddate = data.map(map -> {
                if (!map.containsKey("fielddate")) {
                    // dateTo or smth else?
                    map = Cf.x(map).plus1("fielddate", dateFrom);
                }
                return Cf.x(map).mapValues(v -> {
                    if (v != null && v.getClass().isArray()) {
                        return new String((byte[]) v);
                    }
                    return v;
                });
            });

            logger.debug("Data for date={}, time={}, data: {}", dateForLog,
                    TimeUtils.secondsStringToNow(start), dataWithFielddate);

            statfaceClient.postData(taskInfo.reportName, scale, dataWithFielddate);

            logger.debug("Uploaded to stat for date={}", dateForLog);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    static String prepareYql(String yql, String dateFrom, String dateTo) {
        Pattern pattern = Pattern.compile("\\{\\{/*([^/][^}]*)}}");
        Matcher matcher = pattern.matcher(yql);
        String result = matcher.replaceAll("RANGE(`$1`, `" + dateFrom + "`, `" + dateTo + "`)");
        if (!dateFrom.equals(dateTo) && !result.contains("DefaultOperationWeight")) {
            result = "PRAGMA yt.DefaultOperationWeight=\"0.1\";\n" + result;
        }
        return result;
    }

    private CronTask consTask(YqlStatTaskInfo info) {
        return new CronTask() {
            @Override
            public TaskId id() {
                if (queueName().equals(super.queueName())) {
                    return new TaskId(PREFIX + info.id);
                } else  {
                    throw new IllegalStateException("Not allowed to use another cron queue");
                }
            }

            @Override
            public Schedule cronExpression() {
                // 3 - 6
                int minutesFromIntervalStart =
                        new Random2(info.id.hashCode()).nextInt((int) Duration.standardHours(3).getStandardMinutes());

                int minutes = minutesFromIntervalStart % 60;
                int hours = 3 + minutesFromIntervalStart / 60;

                return ScheduleCron.parse(minutes + " " + hours + " * * *", TimeUtils.EUROPE_MOSCOW_TIME_ZONE);
            }

            @Override
            public Duration timeout() {
                return Duration.standardHours(23);
            }

            @Override
            public void execute(ExecutionContext executionContext) throws Exception {
                executeYqlStat(info.id);
            }
        };
    }
}
