package ru.yandex.chemodan.app.worker2.unlim;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

import org.joda.time.Duration;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.mail.javamail.MimeMessageHelper;

import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.commune.bazinga.scheduler.CronTask;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleCron;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleWithRetry;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author vpronto
 * TODO use freemarker
 */
public class UnlimSizesReportTask extends CronTask {

    private static final Logger logger = LoggerFactory.getLogger(UnlimSizesReportTask.class);

    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd");

    private final DynamicProperty<Integer> fromPadding = new DynamicProperty<>("unlim-sizes.from-padding", 8);
    private final DynamicProperty<Integer> toPadding = new DynamicProperty<>("unlim-sizes.to-padding", 1);
    private final DynamicProperty<String> usersYql = new DynamicProperty<>("unlim-sizes.users-yql",
            "SELECT uid, \n"
                    + "SUM_IF(CAST(_other{\"size\"} AS Int64), _other{\"media_type\"}=\"image\") AS image_size, \n"
                    + "COUNT_IF(_other{\"media_type\"}=\"image\") AS image_count,\n"
                    + "SUM_IF(CAST(_other{\"size\"} AS Int64), (_other{\"media_type\"}=\"image\" AND _other{\"hardlinked\"}=\"1\")) AS image_size_hardlinked, \n"
                    + "COUNT_IF((_other{\"media_type\"}=\"image\" AND _other{\"hardlinked\"}=\"1\")) AS image_count_hardlinked,"
                    + "SUM_IF(CAST(_other{\"size\"} AS Int64), _other{\"media_type\"}=\"video\") AS video_size, \n"
                    + "COUNT_IF(_other{\"media_type\"}=\"video\") AS video_count,\n"
                    + "SUM_IF(CAST(_other{\"size\"} AS Int64), (_other{\"media_type\"}=\"video\" AND _other{\"hardlinked\"}=\"1\")) AS video_size_hardlinked, \n"
                    + "COUNT_IF((_other{\"media_type\"}=\"video\" AND _other{\"hardlinked\"}=\"1\")) AS video_count_hardlinked,"
                    + "SUM(CAST(_other{\"size\"} AS Int64)) AS total_size,\n"
                    + "COUNT(1) as total_count,\n"
                    + "SUM(Coalesce(IF( _other{\"hardlinked\"}=\"1\", CAST(_other{\"size\"} AS Int64), 0))) as total_size_hardlinked,\n"
                    + "COUNT_IF(_other{\"hardlinked\"}=\"1\") as total_count_hardlinked\n"
                    + "FROM RANGE([statbox/ydisk-mpfs-stat-log], [{date_from}], [{date_to}])\n"
                    + "WHERE tskv_format=\"ydisk-mpfs-stat-log-store\"\n"
                    + "AND _other{\"operation_subtype\"} = \"photounlim\"\n"
                    + "AND CAST(_other{\"size\"}  AS Int64) IS NOT NULL\n"
                    + "GROUP BY _other{\"uid\"} AS uid\n"
                    + "ORDER BY {order_by} DESC\n"
                    + "LIMIT 100");

    private final DynamicProperty<String> totalSizesYql = new DynamicProperty<>("unlim-sizes.total-sizes-yql",
            "SELECT \n"
                    + "SUM_IF(CAST(_other{\"size\"} AS Int64), _other{\"media_type\"}=\"image\") as image_size, \n"
                    + "COUNT_IF(_other{\"media_type\"}=\"image\") AS image_count,\n"
                    + "SUM_IF(CAST(_other{\"size\"} AS Int64), (_other{\"media_type\"}=\"image\" AND _other{\"hardlinked\"}=\"1\")) as image_size_hardlinked, \n"
                    + "COUNT_IF((_other{\"media_type\"}=\"image\" AND _other{\"hardlinked\"}=\"1\")) AS image_count_hardlinked,\n"
                    + "SUM_IF(CAST(_other{\"size\"} AS Int64), _other{\"media_type\"}=\"video\") as video_size, \n"
                    + "COUNT_IF(_other{\"media_type\"}=\"video\") AS video_count,\n"
                    + "SUM_IF(CAST(_other{\"size\"} AS Int64), (_other{\"media_type\"}=\"video\" AND _other{\"hardlinked\"}=\"1\")) as video_size_hardlinked, \n"
                    + "COUNT_IF((_other{\"media_type\"}=\"video\" AND _other{\"hardlinked\"}=\"1\")) AS video_count_hardlinked,\n"
                    + "SUM(CAST(_other{\"size\"} AS Int64)) AS total_size,\n"
                    + "COUNT(1) as total_count,\n"
                    + "SUM_IF(CAST(_other{\"size\"} AS Int64), _other{\"hardlinked\"}=\"1\") as total_size_hardlinked,\n"
                    + "COUNT_IF(_other{\"hardlinked\"}=\"1\") as total_count_hardlinked\n"
                    + "FROM RANGE([statbox/ydisk-mpfs-stat-log], [{date_from}], [{date_to}])\n"
                    + "WHERE tskv_format=\"ydisk-mpfs-stat-log-store\"\n"
                    + "AND _other{\"operation_subtype\"} = \"photounlim\"\n"
                    + "AND CAST(_other{\"size\"}  AS Int64) IS NOT NULL");

    private final DynamicProperty<String> usersCountYql = new DynamicProperty<>("unlim-sizes.users-count-yql",
            "SELECT count(1) AS total FROM (\n"
                    + "SELECT uid\n"
                    + "FROM RANGE([statbox/ydisk-mpfs-stat-log], [{date_from}], [{date_to}])\n"
                    + "WHERE tskv_format=\"ydisk-mpfs-stat-log-store\"\n"
                    + "AND _other{\"operation_subtype\"} = \"photounlim\"\n"
                    + "GROUP BY _other{\"uid\"} AS uid\n"
                    + ")\n");

    private final static String tableBody =
            "<tr>"
                    + "<th>UID</th><th>Image size (hardlinked)</th>"
                    + "<th>Image count (hardlinked)</th><th>Video size (hardlinked)</th>"
                    + "<th>Video count (hardlinked)</th><th>Total size (hardlinked)</th>"
                    + "<th>Total count (hardlinked)</th>"
                    + "</tr>";

    private final DynamicProperty<String> template = new DynamicProperty<>("unlim-sizes.template",
            "<div id=\"header\">\n"
                    + "<h2>UnlimPhoto report for {date}</h2>\n"
                    + "<table>"
                    + "     <tr><td>Total count of users:  </td> <td> {total_users}       </td> </tr>"
                    + "     <tr><td>Total count of files (hardlinked):  </td> <td> {total_files_count} ({total_files_count_hardlinked})</td> </tr>"
                    + "     <tr><td>Total size  of files (hardlinked):  </td> <td> {total_files_size}  ({total_files_size_hardlinked})</td> </tr>"
                    + "     <tr><td>Total count of images (hardlinked): </td> <td> {total_images_count} ({total_images_count_hardlinked})</td> </tr>"
                    + "     <tr><td>Total size  of images (hardlinked): </td> <td> {total_images_size} ({total_images_size_hardlinked})</td> </tr>"
                    + "     <tr><td>Total count of videos (hardlinked): </td> <td> {total_videos_count} ({total_videos_count_hardlinked})</td> </tr>"
                    + "     <tr><td>Total size  of videos (hardlinked): </td> <td> {total_videos_size} ({total_videos_size_hardlinked})</td> </tr>"
                    + "</table>"
                    + "</div>"
                    + "<div id=\"content\">"
                    + "<h3>Top 100 users by hardlink:</h3>"
                    + "  <table>"
                    + tableBody
                    + "    {list_by_hardlinked}"
                    + "  </table>"
                    + "<h3>Top 100 users by size:</h3>"
                    + "  <table>"
                    + tableBody
                    + "    {list_by_size}"
                    + "  </table>"
                    + "</div><br><br>");

    private final DynamicProperty<String> link = new DynamicProperty<>("unlim-sizes.link",
            "https://disk-support.yandex-team.ru/info?login=");

    private final DynamicProperty<String> recepients = new DynamicProperty<>("unlim-sizes.recepients",
            "disk-dev@yandex-team.ru,akinfold@yandex-team.ru");

    private final DynamicProperty<String> mailHost = new DynamicProperty<>("unlim-sizes.mail-host",
            "outbound-relay.yandex.net");

    private ExecutorService executors;
    private final String token;
    private final String yqlDatasourceUrl;

    public UnlimSizesReportTask(String token, String yqlDatasourceUrl) {
        this.token = token;
        this.yqlDatasourceUrl = yqlDatasourceUrl;
    }

    @Override
    public Schedule cronExpression() {
        return new ScheduleWithRetry(new ScheduleCron("0 11 * * *",
                ru.yandex.misc.time.TimeUtils.EUROPE_MOSCOW_TIME_ZONE), 3);
    }

    @Override
    public Duration timeout() {
        return Duration.standardHours(2);
    }

    @Override
    public void execute(ExecutionContext executionContext) throws Exception {
        execute(getDate(fromPadding.get()), getDate(toPadding.get()));
    }

    public void execute(String dateFrom, String dateTo) throws Exception {
        try {
            logger.info("Starting building report {} : {}", dateFrom, dateTo);
            executors = Executors.newFixedThreadPool(12);
            CompletableFuture<String> one = runAsync(dateTo, dateTo);
            CompletableFuture<String> all = runAsync(dateTo, dateFrom);
            send(one.get() + all.get(), dateTo);
        } finally {
            executors.shutdown();
        }
    }

    private String gatherForDate(String dateFrom, String dateTo) throws Exception {
        Class.forName("ru.yandex.yql.YqlDriver");
        try (Connection conn = DriverManager.getConnection(yqlDatasourceUrl, "unused", token);
             PreparedStatement totalSizesStmt = conn.prepareStatement(prepare(totalSizesYql, dateFrom, dateTo));
             PreparedStatement usersCountStmt = conn.prepareStatement(prepare(usersCountYql, dateFrom, dateTo));
             PreparedStatement usersStmtByTotalSize = conn.prepareStatement(prepare(usersYql, dateFrom, dateTo)
                     .replace("{order_by}", "total_size"));
             PreparedStatement usersStmtByTotalSizeHardlinked = conn.prepareStatement(prepare(usersYql, dateFrom, dateTo)
                     .replace("{order_by}", "total_size_hardlinked"))
        ) {
            CompletableFuture<ResultSet> totalSizesRsO = runAsync(totalSizesStmt);
            CompletableFuture<ResultSet> usersCountRsO = runAsync(usersCountStmt);
            CompletableFuture<ResultSet> usersRsByTotalSizeO = runAsync(usersStmtByTotalSize);
            CompletableFuture<ResultSet> usersRsByTotalSizeHardlinkedO = runAsync(usersStmtByTotalSizeHardlinked);

            try (ResultSet totalSizesRs = totalSizesRsO.get();
                 ResultSet usersCountRs = usersCountRsO.get();
                 ResultSet usersRsByTotalSize = usersRsByTotalSizeO.get();
                 ResultSet usersRsByTotalSizeHardlinked = usersRsByTotalSizeHardlinkedO.get()
            ) {
                usersCountRs.next();
                totalSizesRs.next();
                String message = template.get().replace("{date}", dateFrom.equals(dateTo) ? dateFrom : dateFrom + " : " + dateTo)
                        .replace("{total_users}", usersCountRs.getString("total"))
                        .replace("{total_files_count}", totalSizesRs.getString("total_count"))
                        .replace("{total_files_count_hardlinked}", totalSizesRs.getString("total_count_hardlinked"))
                        .replace("{total_files_size}", humanReadableByteSize(totalSizesRs.getString("total_size")))
                        .replace("{total_files_size_hardlinked}", humanReadableByteSize(totalSizesRs.getString("total_size_hardlinked")))
                        .replace("{total_images_count}", totalSizesRs.getString("image_count"))
                        .replace("{total_images_count_hardlinked}", totalSizesRs.getString("image_count_hardlinked"))
                        .replace("{total_images_size}", humanReadableByteSize(totalSizesRs.getString("image_size")))
                        .replace("{total_images_size_hardlinked}", humanReadableByteSize(totalSizesRs.getString("image_size_hardlinked")))
                        .replace("{total_videos_count}", totalSizesRs.getString("video_count"))
                        .replace("{total_videos_count_hardlinked}", totalSizesRs.getString("video_count_hardlinked"))
                        .replace("{total_videos_size}", humanReadableByteSize(totalSizesRs.getString("video_size")))
                        .replace("{total_videos_size_hardlinked}", humanReadableByteSize(totalSizesRs.getString("video_size_hardlinked")))
                        .replace("{list_by_size}", getResultBy(usersRsByTotalSize).toString())
                        .replace("{list_by_hardlinked}", getResultBy(usersRsByTotalSizeHardlinked).toString());
                logger.info("Generated report: {}", message);
                return message;
            }

        } catch (Exception e) {
            logger.error("Can't execute yql query", e);
            throw ExceptionUtils.translate(e);
        }
    }

    private String prepare(DynamicProperty<String> yql, String dateFrom, String dateTo) {
        return yql.get().replace("{date_from}", dateTo).replace("{date_to}", dateFrom);
    }


    private CompletableFuture<String> runAsync(String dateFrom, String dateTo) {
        return runAsync(() -> gatherForDate(dateFrom, dateTo));
    }

    private CompletableFuture<ResultSet> runAsync(PreparedStatement statement) {
        return runAsync(() -> statement.executeQuery());
    }

    private <T> CompletableFuture<T> runAsync(SupplierWithException<T> supplier) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                return supplier.get();
            } catch (Exception e) {
                throw ExceptionUtils.translate(e);
            }
        }, executors);
    }

    private StringBuilder getResultBy(ResultSet usersRs) throws SQLException {
        String l = link.get();
        StringBuilder bodyGenerator = new StringBuilder();
        while (usersRs.next()) {
            String uid = usersRs.getString("uid");
            bodyGenerator.append("<tr>")
                    .append("<td><a href='").append(l).append(uid).append("'>").append(uid).append("</a></td>")
                    .append("<td>").append(humanReadableByteSize(usersRs.getString("image_size"))).append(" (")
                    .append(humanReadableByteSize(usersRs.getString("image_size_hardlinked"))).append(")</td>")
                    .append("<td>").append(usersRs.getString("image_count")).append(" (")
                    .append(usersRs.getString("image_count_hardlinked")).append(")</td>")
                    .append("<td>").append(humanReadableByteSize(usersRs.getString("video_size"))).append(" (")
                    .append(humanReadableByteSize(usersRs.getString("video_size_hardlinked"))).append(")</td>")
                    .append("<td>").append(usersRs.getString("video_count")).append(" (")
                    .append(usersRs.getString("video_count_hardlinked")).append(")</td>")
                    .append("<td>").append(humanReadableByteSize(usersRs.getString("total_size"))).append(" (")
                    .append(humanReadableByteSize(usersRs.getString("total_size_hardlinked"))).append(")</td>")
                    .append("<td>").append(usersRs.getString("total_count")).append(" (")
                    .append(usersRs.getString("total_count_hardlinked")).append(")</td>")
                    .append("</tr>");
        }
        return bodyGenerator;
    }

    public void send(String message, String date) throws MessagingException {
        String[] to = recepients.get().split(",");
        logger.info("About to send report to {}", Arrays.toString(to));
        JavaMailSenderImpl mailSender = new JavaMailSenderImpl();
        mailSender.setHost(mailHost.get());
        mailSender.setPort(25);
        MimeMessage mimeMessage = mailSender.createMimeMessage();
        MimeMessageHelper smm = new MimeMessageHelper(mimeMessage, true);
        smm.setText(message, true);
        smm.setFrom("yandex_personal_cloud_dev_java@yandex-team.ru");
        smm.setSubject("UnlimPhoto report for " + date);
        smm.setTo(to);
        mailSender.send(mimeMessage);
        new RetryManager().withRetryPolicy(3, 100).run(() ->
                mailSender.send(mimeMessage));
    }

    public static String humanReadableByteSize(String size) {
        if (StringUtils.isEmpty(size)) {
            return "";
        }
        long bytes = Long.parseLong(size);
        return DataSize.prettyString(bytes);
    }

    private String getDate(int padding) {
        return LocalDate.now().minusDays(padding).toString(formatter);
    }

    private interface SupplierWithException<T> {
        T get() throws Exception;
    }
}
