package ru.yandex.chemodan.app.worker2.wakeup;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.worker2.xiva.XivaPushSender;
import ru.yandex.chemodan.util.date.LocalDateRange;
import ru.yandex.chemodan.util.yt.AbstractTablePath;
import ru.yandex.chemodan.util.yt.DateRangeTablePath;
import ru.yandex.chemodan.util.yt.TableBatchExecutor;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.chemodan.util.yt.YtSpecBuilder;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.operations.specs.MapReduceSpec;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.MoscowTime;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class WakeUpPushManager {
    private static final Logger logger = LoggerFactory.getLogger(WakeUpPushManager.class);

    private static final YPath ROOT_PATH = YPath.simple("//home/mpfs-stat");

    private static final YPath TESTING_ROOT_PATH = ROOT_PATH.child("testing");

    private final YtHelper yt;

    private final BazingaTaskManager bazingaTaskManager;

    private final XivaPushSender pushSender;

    private final YPath rootPath;

    private final YPath mpfsAccessLogPath = YPath.simple("//statbox/ydisk-mpfs-access-log");

    private final int removeDataOlderThanDays;

    private final int aggregateDataForLastDays;

    private final Duration schedulingBatchSendDuration;

    private final int schedulingBatchSize;

    private final boolean sendDryRun;

    private final DynamicProperty<Boolean> batchSendEnabled;

    public WakeUpPushManager(YtHelper yt, BazingaTaskManager bazingaTaskManager, XivaPushSender pushSender,
            int removeDataOlderThanDays, int aggregateDataForLastDays, Duration schedulingBatchSendDuration,
            int schedulingBatchSize, boolean sendDryRun, DynamicProperty<Boolean> batchSendEnabled)
    {
        this.yt = yt;
        this.bazingaTaskManager = bazingaTaskManager;
        this.pushSender = pushSender;
        this.rootPath = getRootPath();
        this.removeDataOlderThanDays = removeDataOlderThanDays;
        this.aggregateDataForLastDays = aggregateDataForLastDays;
        this.schedulingBatchSendDuration = schedulingBatchSendDuration;
        this.schedulingBatchSize = schedulingBatchSize;
        this.sendDryRun = sendDryRun;
        this.batchSendEnabled = batchSendEnabled;
    }

    public void updateIosUsers() {
        LocalDate today = MoscowTime.today();

        removeOldTables(today);

        LocalDateRange targetRange = LocalDateRange.daysUpTo(aggregateDataForLastDays - 1, today.minusDays(1));
        ListF<YPath> inputTables = getInputTables(targetRange);

        if (inputTables.isEmpty()) {
            return;
        }

        MapReduceSpec mapReduceSpec = YtSpecBuilder.python(getScriptPath())
                .inputTables(inputTables)
                .outputTable(getOutputPath().child(targetRange.toString()))
                .map("map")
                .params(targetRange.getStart())
                .reduce("reduce")
                .by(Cf.list("uid"))
                .build();
        yt.uploadScriptsWithRetries(this.getClass(), Cf.list(getScriptPath()));
        yt.getWithRetries(() -> yt.operations().mapReduceAndGetOp(mapReduceSpec));
    }

    private void removeOldTables(LocalDate today) {
        LocalDate cleanBeforeDate = today.minusDays(removeDataOlderThanDays);
        Tuple2<ListF<DateRangeTablePath>, ListF<DateRangeTablePath>> deleteAndKeepTables =
                yt.getDateRangePaths(getOutputPath())
                        .partition(rangePath -> rangePath.getEndDate().isBefore(cleanBeforeDate));
        boolean noActiveTables = deleteAndKeepTables.get2()
                .filter(AbstractTablePath::isNotEmpty)
                .isEmpty();
        if (noActiveTables) {
            return;
        }

        deleteAndKeepTables.get1()
                .forEach(path -> yt.remove(path.getPath()));
    }

    private ListF<YPath> getInputTables(LocalDateRange targetRange) {
        ListF<DateRangeTablePath> overlappingExistingRanges = yt.getDateRangePaths(getOutputPath())
                .filter(rangePath -> rangePath.overlaps(targetRange))
                .filter(AbstractTablePath::isNotEmpty);
        Option<DateRangeTablePath> supersetOfTgtRangeO =
                overlappingExistingRanges.find(rangePath -> rangePath.includes(targetRange));
        if (supersetOfTgtRangeO.isPresent()) {
            logger.info("Superset of target range already exists: {} includes {}",
                    supersetOfTgtRangeO.get().getName(), targetRange);
            return Cf.list();
        }

        return targetRange.toSet()
                .minus(LocalDateRange.collectLocalDateSet(overlappingExistingRanges))
                .map(date -> yt.getDailyTablePath(mpfsAccessLogPath, date))
                .filter(AbstractTablePath::existsAndNotEmpty)
                .map(AbstractTablePath::getPath)
                .plus(overlappingExistingRanges.map(AbstractTablePath::getPath));
    }

    private YPath getScriptPath() {
        return rootPath.child("scripts").child("update_ios_users.py");
    }

    private YPath getOutputPath() {
        return rootPath.child("iosUsers");
    }

    public void scheduleWakeUpPushSchedulers() {
        Option<YPath> pathO = getIosUsersPath();
        if (!pathO.isPresent()) {
            logger.warn("No table with latest iOS users");
            return;
        }

        YPath path = pathO.get();
        final double rowCount = yt.getRowCount(path);
        final long batchesCount = (long) Math.ceil(rowCount / schedulingBatchSize);
        final long schedulingBatchDelayInMillis = schedulingBatchSendDuration.getMillis() / batchesCount;
        Instant startTime = Instant.now();
        new TableBatchExecutor(yt, path, schedulingBatchSize)
                .executeByIndexes((startIndex, endIndex) ->
                        scheduleWakeUpPushScheduler(path, startIndex, endIndex,
                                startTime.plus(schedulingBatchDelayInMillis * (startIndex / schedulingBatchSize))
                        )
                );
    }

    private Option<YPath> getIosUsersPath() {
        return yt.getDateRangePaths(getOutputPath())
                .filter(AbstractTablePath::isNotEmpty)
                .sortedByDesc(DateRangeTablePath::getEndDate)
                .firstO()
                .map(AbstractTablePath::getPath);
    }

    private void scheduleWakeUpPushScheduler(YPath path, Long startIndex, Long endIndex, Instant date) {
        bazingaTaskManager.schedule(new ScheduleWakeUpPushRangeTask(path, startIndex, endIndex), date);
    }

    public void scheduleWakeUpPushTasksOrSendBatch(YPath path) {
        if (!batchSendEnabled.get()) {
            yt.runWithRetries(() -> yt.tables().read(path, YTableEntryTypes.bender(UidAndLastSeen.class),
                    this::scheduleWakeUpPushTask));
        } else {
            yt.runWithRetries(() -> yt.tables().read(path, YTableEntryTypes.bender(UidAndLastSeen.class), it -> {
                final ListF<UidAndLastSeen> uidAndLastSeenList = Cf.x(it).toList();
                final ListF<PassportUid> passportUidList = uidAndLastSeenList.filterMap(UidAndLastSeen::getPassportUid);
                batchSendWakeUpPushes(passportUidList);
                return null;
            }));
        }
    }

    private void scheduleWakeUpPushTask(UidAndLastSeen uidAndLastSeen) {
        final Option<PassportUid> passportUid = uidAndLastSeen.getPassportUid();
        if (passportUid.isPresent()) {
            bazingaTaskManager.schedule(new SendWakeUpPushTask(passportUid.get()));
        }
    }

    public void sendWakeUpPushes(PassportUid uid) {
        logger.info("Sending wakeup push to uid#{}", uid);
        if (sendDryRun) {
            return;
        }
        pushSender.sendWakeUpPushes(uid);
    }

    public void batchSendWakeUpPushes(ListF<PassportUid> uidList) {
        logger.info("Sending batch of wakeup pushes of size {} starting from uid#{}", uidList.size(), uidList.first());
        if (sendDryRun) {
            return;
        }
        pushSender.sendWakeUpPushes(uidList);
    }

    private static YPath getRootPath() {
        return EnvironmentType.getActive() == EnvironmentType.PRODUCTION ? ROOT_PATH : TESTING_ROOT_PATH;
    }

    @BenderBindAllFields
    static final class UidAndLastSeen extends DefaultObject {
        final String uid;

        @BenderPart(name = "last_seen", strictName = true)
        final String lastSeen;

        UidAndLastSeen(String uid, String lastSeen) {
            this.uid = uid;
            this.lastSeen = lastSeen;
        }

        public Option<PassportUid> getPassportUid() {
            try {
                return Option.of(PassportUid.cons(Long.parseLong(uid)));
            } catch (NumberFormatException ex) {
                logger.error("Could not parse passport UID from {}", uid);
                return Option.empty();
            }
        }
    }
}
