package ru.yandex.chemodan.app.worker2.wakeup;

import net.jodah.failsafe.FailsafeException;
import org.apache.http.Header;
import org.junit.Test;
import org.mockito.ArgumentCaptor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.xiva.XivaEvent;
import ru.yandex.chemodan.xiva.XivaResponseHeaders;
import ru.yandex.chemodan.xiva.XivaSendResponse;
import ru.yandex.chemodan.xiva.XivaSendResult;
import ru.yandex.chemodan.xiva.XivaSingleTokenClient;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypeUtils;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.test.Assert;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class WakeUpPushManagerTest {

    private static final ListF<WakeUpPushManager.UidAndLastSeen> uidAndLastSeenList = Cf.list(
            new WakeUpPushManager.UidAndLastSeen("123", "2018-04-26"),
            new WakeUpPushManager.UidAndLastSeen("234", "2018-04-27"),
            new WakeUpPushManager.UidAndLastSeen("345", "2018-04-28"));

    @Test
    public void parseUidAndDate() {
        BenderJsonParser<WakeUpPushManager.UidAndLastSeen> parser =
                Bender.jsonParser(
                        WakeUpPushManager.UidAndLastSeen.class,
                        YTableEntryTypeUtils.getDefaultBenderConfiguration()
                );
        Assert.equals(uidAndLastSeenList.first(), parser.parseJson("{\"uid\":\"123\",\"last_seen\":\"2018-04-26\"}"));
    }

    @Test
    public void testGetPassportUid() {
        final PassportUid passportUid = PassportUid.cons(123);
        Assert.equals(uidAndLastSeenList.first().getPassportUid().get(), passportUid);
    }

    @Test
    public void testScheduleWakeUpPushTasks() {
        final XivaSingleTokenClient xivaSingleTokenClient = mock(XivaSingleTokenClient.class);
        final WakeUpPushManager wakeUpPushManager =
                WakeUpPushManagerTestHelper.mockDependeniesForWakeUpManager(xivaSingleTokenClient, uidAndLastSeenList,
                        false);
        final ArgumentCaptor<XivaEvent> eventCaptor = ArgumentCaptor.forClass(XivaEvent.class);
        when(xivaSingleTokenClient.send(eventCaptor.capture()))
                .thenReturn(new XivaSendResponse(XivaResponseHeaders.cons(new Header[0]),
                        Cf.<ListF<XivaSendResult>>list().plus1(Cf.list(new XivaSendResult(200, "OK")))));
        YPath path = YPath.simple("//some/path");
        wakeUpPushManager.scheduleWakeUpPushTasksOrSendBatch(path);
        final ListF<XivaEvent> xivaEventList = Cf.x(eventCaptor.getAllValues());
        Assert.equals(xivaEventList.size(), 3);
        Assert.isTrue(xivaEventList.forAll(event -> event.getBody().isPresent()));
    }

    @Test
    public void testSendBatch() {
        final XivaSingleTokenClient xivaSingleTokenClient = mock(XivaSingleTokenClient.class);
        final WakeUpPushManager wakeUpPushManager =
                WakeUpPushManagerTestHelper.mockDependeniesForWakeUpManager(xivaSingleTokenClient, uidAndLastSeenList,
                        true);
        final ArgumentCaptor<XivaEvent> eventCaptor = ArgumentCaptor.forClass(XivaEvent.class);
        when(xivaSingleTokenClient.send(eventCaptor.capture()))
                .thenReturn(new XivaSendResponse(XivaResponseHeaders.cons(new Header[0]),
                        Cf.<ListF<XivaSendResult>>list().plus1(Cf.list(new XivaSendResult(200, "OK")))));
        YPath path = YPath.simple("//some/path");
        wakeUpPushManager.scheduleWakeUpPushTasksOrSendBatch(path);
        final ListF<XivaEvent> xivaEventList = Cf.x(eventCaptor.getAllValues());
        Assert.equals(xivaEventList.size(), 1);
        Assert.isTrue(xivaEventList.forAll(event -> event.getBody().isPresent()));
    }

    @Test
    public void testLogResponseError() {
        final XivaSingleTokenClient xivaSingleTokenClient = mock(XivaSingleTokenClient.class);
        final WakeUpPushManager wakeUpPushManager =
                WakeUpPushManagerTestHelper.mockDependeniesForWakeUpManager(xivaSingleTokenClient, uidAndLastSeenList,
                        false);
        final ArgumentCaptor<XivaEvent> eventCaptor = ArgumentCaptor.forClass(XivaEvent.class);
        when(xivaSingleTokenClient.send(eventCaptor.capture()))
                .thenReturn(new XivaSendResponse(XivaResponseHeaders.cons(new Header[0]),
                        Cf.<ListF<XivaSendResult>>list().plus1(Cf.list(new XivaSendResult(500, "Internal server error")))));
        YPath path = YPath.simple("//some/path");
        Assert.assertThrows(() -> wakeUpPushManager.scheduleWakeUpPushTasksOrSendBatch(path),
                FailsafeException.class, e -> e.getCause().getMessage().startsWith("Error while sending push to uid"));
    }

    @Test
    public void testLogBatchResponseError() {
        final XivaSingleTokenClient xivaSingleTokenClient = mock(XivaSingleTokenClient.class);
        final WakeUpPushManager wakeUpPushManager =
                WakeUpPushManagerTestHelper.mockDependeniesForWakeUpManager(xivaSingleTokenClient, uidAndLastSeenList,
                        true);
        final ArgumentCaptor<XivaEvent> eventCaptor = ArgumentCaptor.forClass(XivaEvent.class);
        when(xivaSingleTokenClient.send(eventCaptor.capture()))
                .thenReturn(new XivaSendResponse(XivaResponseHeaders.cons(new Header[0]),
                        Cf.<ListF<XivaSendResult>>list().plus1(Cf.list(new XivaSendResult(500, "Internal server error")))));
        YPath path = YPath.simple("//some/path");
        Assert.assertThrows(() -> wakeUpPushManager.scheduleWakeUpPushTasksOrSendBatch(path),
                FailsafeException.class, e -> e.getCause().getMessage().startsWith("Error while sending push to subscription id"));
    }
}
