import cyson
import itertools
import sys
from cyson import YsonEntity


def do_decode(fd):
    rows = cyson.list_fragments(cyson.InputStream.from_fd(fd))
    for row in rows:
        yield row


def do_map(rows, writer, min_last_seen):
    for row in rows:
        if isinstance(row, YsonEntity):
            continue

        if 'uid' not in row:
            continue

        if 'ycrid' in row:
            if not row['ycrid'].startswith('ios-'):
                continue
            result = {'uid': row['uid'], 'last_seen': row['timestamp'][:10]}
        else:
            result = row

        if result.get('last_seen') >= min_last_seen:
            writer.write(result)


def do_reduce(rows, writer):
    prev_row = {}

    for row in itertools.chain(rows, ({},)):
        if prev_row.get('uid') != row.get('uid'):
            if prev_row:
                writer.write(prev_row)
            prev_row = row

        if prev_row.get('last_seen', '') < row.get('last_seen', ''):
            prev_row = row


if __name__ == '__main__':
    stream = do_decode(sys.stdin.fileno())
    actions = {'map': do_map, 'reduce': do_reduce}
    if len(sys.argv) < 2 or sys.argv[1] not in actions:
        print >> sys.stderr, 'Please, specify proper map or reduce',
        print >> sys.stderr, 'command to execute'
        sys.exit(1)

    cyson_writer = cyson.Writer(
        cyson.OutputStream.from_fd(sys.stdout.fileno()),
        format='binary',
        mode='list_fragment',
    )
    cyson_writer.begin_stream()
    actions[sys.argv[1]](stream, cyson_writer, *sys.argv[2:])
    cyson_writer.end_stream()
