package ru.yandex.chemodan.app.worker2.xiva;

import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.RetryPolicy;
import org.jetbrains.annotations.NotNull;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.xiva.DiskXivaServices;
import ru.yandex.chemodan.xiva.XivaClient;
import ru.yandex.chemodan.xiva.XivaEvent;
import ru.yandex.chemodan.xiva.XivaEventRecipient;
import ru.yandex.chemodan.xiva.XivaPushMapBody;
import ru.yandex.chemodan.xiva.XivaPushService;
import ru.yandex.chemodan.xiva.XivaSendResponse;
import ru.yandex.chemodan.xiva.XivaSendResult;
import ru.yandex.chemodan.xiva.XivaSingleTokenClient;
import ru.yandex.chemodan.xiva.XivaSubscription;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Level;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class XivaPushSender {
    private static final Logger logger = LoggerFactory.getLogger(XivaPushSender.class);

    public static final RetryPolicy RETRY_POLICY = new RetryPolicy()
            .withMaxRetries(3)
            .withDelay(100, TimeUnit.MILLISECONDS);

    private final ThreadPoolExecutor executor;

    private final XivaSingleTokenClient xivaClient;

    public XivaPushSender(XivaClient xivaClient, int maxPoolSize) {
        this.xivaClient = xivaClient.toSingleTokenClient(DiskXivaServices.DISK_JSON);
        this.executor = new ThreadPoolExecutor(0, maxPoolSize, 10, TimeUnit.MINUTES,
                new LinkedBlockingQueue<>()
        );
    }

    @SuppressWarnings("unused")
    public void sendWakeUpPushes(String path) {
        new File2(path)
                .asReaderSource()
                .forEachLine(uidStr -> {
                    logger.info("Scheduling wake up push sending for uid#{}", uidStr);
                    executor.execute(() -> parseUidAndSendPush(uidStr));
                });
    }

    private void parseUidAndSendPush(String uidStr) {
        PassportUid uid;
        try {
            uid = new PassportUid(Long.parseLong(uidStr));
        } catch (NumberFormatException ex) {
            logger.error("Error while parsing uid from string = {}", uidStr);
            return;
        }

        sendWakeUpPushes(uid);
    }

    @SuppressWarnings("unused")
    public void setMaxPoolSize(int maxPoolSize) {
        executor.setMaximumPoolSize(maxPoolSize);
    }

    public void sendWakeUpPushes(PassportUid uid) {
        ListF<XivaSubscription> subscriptions = xivaClient.list(uid, "disk-json");
        if (subscriptions.isEmpty()) {
            logger.warn("No APNS subscriptions found for user#{}", uid);
            return;
        }

        subscriptions.filter(subscription -> subscription.getPlatform().isSome(XivaPushService.APNS.value()))
                .forEach(subscription -> sendWakeUpPushWithRetries(uid, subscription));
    }

    public void sendWakeUpPushes(ListF<PassportUid> uidList) {
        final MapF<PassportUid, ListF<XivaSubscription>> uidSubscriptionListMap =
                Cf.toMap(uidList.map(this::listSubscriptions));
        ListF<MapF<PassportUid, XivaSubscription>> uidSubscriptionMapList =
                splitToMapsWithOneSubscriptionForEveryUid(uidSubscriptionListMap);
        uidSubscriptionMapList
                .forEach(this::sendWakeUpPushWithRetries); //list of maps, each map will be sent separately
    }

    //all subscriptions for one uid will be distributed to different maps because of Xiva API batch send limitations
    static ListF<MapF<PassportUid, XivaSubscription>> splitToMapsWithOneSubscriptionForEveryUid(
            MapF<PassportUid, ListF<XivaSubscription>> uidSubscriptionListMap)
    {
        final ListF<MapF<PassportUid, XivaSubscription>> uidSubscriptionMapList = Cf.arrayList();
        uidSubscriptionListMap.forEach((uid, subscriptionList) -> {
            for (int i = 0; i < subscriptionList.size(); i++) {
                MapF<PassportUid, XivaSubscription> uidSubscriptionMap;
                if (uidSubscriptionMapList.size() > i) {
                    uidSubscriptionMap = uidSubscriptionMapList.get(i);
                } else {
                    uidSubscriptionMap = Cf.hashMap();
                    uidSubscriptionMapList.add(uidSubscriptionMap);
                }
                uidSubscriptionMap.put(uid, subscriptionList.get(i));
            }
        });
        return uidSubscriptionMapList;
    }

    @NotNull
    private Tuple2<PassportUid, ListF<XivaSubscription>> listSubscriptions(PassportUid uid) {
        final ListF<XivaSubscription> subscriptions = xivaClient.list(uid, "disk-json");
        if (subscriptions.isEmpty()) {
            logger.warn("No APNS subscriptions found for user#{}", uid);
        }
        final ListF<XivaSubscription> filteredSubscriptions =
                subscriptions.filter(subscription -> subscription.getPlatform().isSome(XivaPushService.APNS.value()));
        return Tuple2.tuple(uid, filteredSubscriptions);
    }

    private void sendWakeUpPushWithRetries(PassportUid uid, XivaSubscription subscription) {
        Failsafe.with(RETRY_POLICY)
                .run(() -> sendWakeUpPush(uid, subscription));
    }

    private void sendWakeUpPushWithRetries(MapF<PassportUid, XivaSubscription> uidSubscriptionMap) {
        if (uidSubscriptionMap.isNotEmpty()) {
            Failsafe.with(RETRY_POLICY).run(() -> sendBatchWakeUpPush(uidSubscriptionMap));
        }
    }

    private void sendWakeUpPush(PassportUid uid, XivaSubscription subscription) {
        XivaSendResponse response = xivaClient.send(buildWakeUpEvent(uid, subscription));
        if (response.results.isEmpty()) {
            logger.warn("Unknown state while sending push for uid#{} with subscription#{}", uid,
                    subscription.getId());
        } else {
            response.results.firstO()
                    .filterMap(ListF::firstO)
                    .ifPresent(sendResult -> logResponse(uid, subscription, sendResult));
        }
    }

    private void logResponse(PassportUid uid, XivaSubscription subscription, XivaSendResult sendResult) {
        if (sendResult.isOk()) {
            logger.log(Level.INFO, "Successfully sent push for uid#{} with subscription#{}: status={}, body={}",
                    uid, subscription.getId(), sendResult.status, sendResult.body);
        } else {
            final String message =
                    String.format("Error while sending push to uid#%s: status=%d, body=%s", uid, sendResult.status,
                            sendResult.body);
            logger.log(Level.ERROR, message);
            throw new IllegalStateException(message);
        }
    }

    private static XivaEvent buildWakeUpEvent(PassportUid uid, XivaSubscription subscription) {
        return buildWakeUpEvent(uid, subscription, Random2.threadLocal().nextAlnum(64));
    }

    static XivaEvent buildWakeUpEvent(PassportUid uid, XivaSubscription subscription, String random) {
        String sessionId = String.format("%s:%s", uid, random);
        return new XivaEvent(XivaEventRecipient.passportSubscription(uid, subscription.getId()), "wake_up")
                .withBody(buildWakeUpPushBody(sessionId));
    }

    private static XivaPushMapBody buildWakeUpPushBody(String sessionId) {
        return XivaPushMapBody.consStr(
                Cf.<String, Object>linkedHashMap()
                        .plus1("root",
                                Cf.linkedHashMap()
                                        .plus1("session_id", sessionId)
                                        .plus1("tag", "wake_up")
                        )
        );
    }

    private void sendBatchWakeUpPush(MapF<PassportUid, XivaSubscription> uidSubscriptionMap) {
        XivaSendResponse response = xivaClient.send(buildBatchWakeUpEvent(uidSubscriptionMap));
        final PassportUid firstPassportUid = firstEntry(uidSubscriptionMap).getKey();
        final int batchSize = uidSubscriptionMap.size();
        final ListF<ListF<XivaSendResult>> results = response.results;
        if (results.isEmpty()) {
            logger.warn("Unknown state while sending push for batch starting from uid#{} with batch size {}",
                    firstPassportUid, batchSize);
        } else {
            if (results.forAll(l -> l.forAll(XivaSendResult::isOk))) {
                logger.log(Level.INFO,
                        "Successfully batch sent pushes starting from uid#{} with batch size {}",
                        firstPassportUid, batchSize);
            } else {
                logBatchResponseErrors(results);
            }
        }
    }

    private void logBatchResponseErrors(ListF<ListF<XivaSendResult>> resultList) {
        resultList.stream()
                .flatMap(CollectionF::stream)
                .filter(result -> !result.isOk())
                .map(result -> String.format("Error while sending push to subscription id#%s: status=%d, body=%s",
                        result.subscriptionId, result.status, result.body))
                .forEach(message -> {
                    logger.log(Level.ERROR, message);
                    throw new IllegalArgumentException(message);
                });
    }

    private static XivaEvent buildBatchWakeUpEvent(MapF<PassportUid, XivaSubscription> uidSubscriptionMap) {
        return buildBatchWakeUpEvent(uidSubscriptionMap, Random2.threadLocal().nextAlnum(64));
    }

    static XivaEvent buildBatchWakeUpEvent(MapF<PassportUid, XivaSubscription> uidSubscriptionMap, String random)
    {
        String sessionId = String.format("%s:%s", firstEntry(uidSubscriptionMap).getKey(), random);
        return new XivaEvent(XivaEventRecipient.multipleSubscriptions(uidSubscriptionMap), "wake_up")
                .withBody(buildWakeUpPushBody(sessionId));
    }

    private static Map.Entry<PassportUid, XivaSubscription> firstEntry(
            MapF<PassportUid, XivaSubscription> uidSubscriptionMap)
    {
        return uidSubscriptionMap.entrySet().iterator().next();
    }
}
