# encoding: UTF-8

import typing as t

from appcore.web.plugin import JsonWebErrorMixin


class SecurityError(Exception):
    pass


class AccessError(SecurityError):
    pass


class UnauthenticatedAccessError(AccessError, JsonWebErrorMixin):
    def _get_status_code(self):
        return 401


class UnauthorizedAccessError(AccessError, JsonWebErrorMixin):
    def _get_status_code(self):
        return 403


class UnauthenticatedClientError(UnauthenticatedAccessError):
    def __init__(self):
        super(UnauthenticatedClientError, self).__init__(
            'Unauthenticated client.'
        )


class NoRequiredScopesError(UnauthorizedAccessError):
    def __init__(self, required_scopes, requires_all):
        super(NoRequiredScopesError, self).__init__(
            'Client has no required scopes.',
        )

        self.required_scopes = required_scopes  # type: t.Iterable
        self.requires_all = requires_all  # type: bool

    def _get_details(self):
        return dict(
            required_scopes=list(self.required_scopes),
            requires_all=self.requires_all,
        )


class UnauthenticatedOrgError(UnauthenticatedAccessError):
    def __init__(self):
        super(UnauthenticatedOrgError, self).__init__(
            'Unauthenticated organization.'
        )


class UnauthenticatedUserError(UnauthenticatedAccessError):
    def __init__(self):
        super(UnauthenticatedUserError, self).__init__(
            'Unauthenticated user.'
        )


class NoRequiredPermsError(UnauthorizedAccessError):
    def __init__(self, required_perms, requires_all):
        super(NoRequiredPermsError, self).__init__(
            'User has no required perms.',
        )

        self.required_perms = required_perms  # type: t.Iterable
        self.requires_all = requires_all  # type: bool

    def _get_details(self):
        return dict(
            required_perms=list(self.required_perms),
            requires_all=self.requires_all,
        )
