# encoding: UTF-8

import logging

import flask
import typing as t
import ylog.context

from appcore.security.provider import AuthProvider
from appcore.security.dto import SecurityContextSchema
from appcore.security.model import AuthIdentity
from appcore.security.model import RemoteIdentity
from appcore.security.model import SecurityContext
from appcore.security.identifying import IdentifyingStrategy

_log = logging.getLogger(__name__)


class SecurityManager(object):
    def __init__(self, identifying_strategy):
        self._identifying_strategy = identifying_strategy  # type: IdentifyingStrategy
        self.auth_providers = []  # type: list[AuthProvider]

    def identify_remote(self, request):
        # type: (flask.Request) -> RemoteIdentity

        return self._identifying_strategy.identify_remote(request)

    def authorize(self, request, remote_identity):
        # type: (flask.Request, RemoteIdentity) -> AuthIdentity

        for provider in self.auth_providers:
            auth_identity = provider.authenticate_request(
                request,
                remote_identity,
            )
            if auth_identity is not None:
                return auth_identity

        return AuthIdentity(
            client_id=None,
            client_scopes=set(),
            user_id=None,
        )

    @staticmethod
    def _make_ctx_dto(ctx):
        # type: (SecurityContext) -> t.Any

        ctx_schema = SecurityContextSchema()
        ctx_dto, errors = ctx_schema.dump(ctx)
        if errors:
            with ylog.context.log_context(errors=errors):
                _log.error('Failed to serialize SecurityContext object')

            return None
        else:
            return ctx_dto

    def create_security_context(self, request):
        # type: (flask.Request) -> SecurityContext

        remote_identity = self.identify_remote(request)
        auth_identity = self.authorize(request, remote_identity)
        ctx = SecurityContext(remote_identity, auth_identity)

        ctx_dto = self._make_ctx_dto(ctx)
        if ctx_dto:
            ylog.context.put_to_context('security', ctx_dto)

        if auth_identity.is_authenticated_client:
            _log.info('Client authenticated')

        if auth_identity.is_authenticated_user:
            _log.info('User authenticated')

        return ctx
