# encoding: UTF-8

import logging

import flask
import werkzeug.local

from appcore.injection import inject
from appcore.plugin import PluginBase
from appcore.security.model import AuthIdentity
from appcore.security.model import RemoteIdentity
from appcore.security.model import SecurityContext

_log = logging.getLogger(__name__)

G_SECURITY_CONTEXT = 'security_context'


class SecurityPlugin(PluginBase):
    def __init__(
            self,
            security_manager_name='security_manager',
            app=None
    ):
        self.security_manager_name = security_manager_name
        super(SecurityPlugin, self).__init__(app)

    def _do_init_app(self):
        self.app.before_request(self._authenticate_request)

    def _authenticate_request(self):
        security_manager = inject(self.security_manager_name)
        ctx = security_manager.create_security_context(flask.request)
        setattr(flask.g, G_SECURITY_CONTEXT, ctx)


def _get_security_context():
    # type: () -> SecurityContext

    return getattr(flask.g, G_SECURITY_CONTEXT, None)


security_context = werkzeug.local.LocalProxy(
    _get_security_context,
)  # type: SecurityContext


def _get_remote_identity():
    # type: () -> RemoteIdentity

    return security_context.remote_identity


remote_identity = werkzeug.local.LocalProxy(
    _get_remote_identity
)  # type: RemoteIdentity


def _get_auth_identity():
    # type: () -> AuthIdentity

    return security_context.auth_identity


auth_identity = werkzeug.local.LocalProxy(
    _get_auth_identity
)  # type: AuthIdentity
