# encoding: UTF-8

import struct
import time

import lmdb
import numpy as np
import pandas as pd

from appcore.unistat.stat import AggType
from appcore.unistat.stat import Stat

DEFAULT_PATH = '/tmp/unistat'
DEFAULT_MAX_SIZE = 512 * (1024 ** 2)  # 512MB
DEFAULT_METASYNC = True
DEFAULT_SYNC = True


class UnistatAgent(object):
    _RECORD_STRUCT = struct.Struct('dd')

    def __init__(self):
        self._env = None  # type: lmdb.Environment
        self._db = None

        self.setup()

    def __del__(self):
        self.close()

    def close(self):
        if self._env:
            self._env.close()
            self._env = None
            self._db = None

    def setup(
            self,
            path=DEFAULT_PATH,
            max_size=DEFAULT_MAX_SIZE,
            metasync=DEFAULT_METASYNC,
            sync=DEFAULT_SYNC
    ):
        """
        Настройка базы данных агента.
        """

        self.close()

        self._env = lmdb.Environment(
            path,
            map_size=max_size,
            metasync=metasync,
            sync=sync,
            readahead=False,
            meminit=False,
            max_dbs=1
        )
        self._db = self._env.open_db('metrics', dupsort=True, dupfixed=True)

    def __put_value(self, cursor, stat, ts, value):
        cursor.put(
            stat.unistat_name,
            self._RECORD_STRUCT.pack(ts, value)
        )

    def put(self, stat, value):
        """
        Сохраняет значение метрики.
        """

        with self._env.begin(write=True) as txn:
            cursor = txn.cursor(self._db)
            self.__put_value(cursor, stat, time.time(), value)

    def put_all(self, stat_values):
        """
        Как put, но сохраняет несколько значений в одной транзакции.

        Например:

        unistat_agent.put_all([
            (stat1, 0.2),
            (stat1, 0.2),
            (stat2, 1342.5),
        ])
        """

        with self._env.begin(write=True) as txn:
            cursor = txn.cursor(self._db)
            ts = time.time()
            for stat, value in stat_values:
                self.__put_value(cursor, stat, ts, value)

    def aggregate_stats(self, stats):
        """
        Возвращает список значений для заданных метрик,
        совместимый с форматом unistat-ручки.

        Независимо от списка запрашиваемых метрик, база данных значений
        сбрасывается полностью (защита от распухания).
        """

        stats = {
            stat.unistat_name: stat
            for stat in stats
        }
        results = []

        items = []
        with self._env.begin(write=True) as txn:
            cursor = txn.cursor(self._db)
            cursor.first()
            key, data = cursor.item()
            while cursor.delete():
                timestamp, value = self._RECORD_STRUCT.unpack(data)
                items.append((key, timestamp, value))
                key, data = cursor.item()
        df = pd.DataFrame(data=items, columns=('name', 'ts', 'value'))
        df['ts'] = pd.to_datetime(df['ts'], unit='s')

        for name, group in df.groupby('name'):  # type: basestring, pd.DataFrame
            stat = stats.pop(name, None)  # type: Stat

            if stat is None:
                continue

            agg_type = stat.hagg

            if agg_type == AggType.aver:
                agg_value = group['value'].mean()
                agg_value = float(agg_value)

            elif agg_type == AggType.max:
                agg_value = group['value'].max()
                agg_value = float(agg_value)

            elif agg_type == AggType.min:
                agg_value = group['value'].min()
                agg_value = float(agg_value)

            elif agg_type == AggType.summ:
                agg_value = group['value'].sum()
                agg_value = float(agg_value)

            elif agg_type == AggType.summone:
                if len(group):
                    agg_value = group['value'].sum()
                    agg_value = float(agg_value)
                else:
                    agg_value = None

            elif agg_type == AggType.trnsp:
                idx = group['ts'].idxmax()
                agg_value = group['value'][idx]
                agg_value = float(agg_value)

            elif agg_type == AggType.counter:
                agg_value = len(group)

            elif agg_type == AggType.hgram:
                values, bins = np.histogram(
                    group['value'],
                    bins=stat.bins,
                )
                agg_value = map(list, zip(bins, values))

            else:
                msg = 'Aggregation for %s not implemented' % agg_type
                raise NotImplementedError(msg)

            results.append([stat.unistat_name, agg_value])

        for name, stat in stats.iteritems():
            agg_type = stat.hagg
            if agg_type in {AggType.aver, AggType.max, AggType.min,
                            AggType.summone, AggType.trnsp}:
                continue
            elif agg_type in {AggType.summ, AggType.counter}:
                agg_value = 0
            elif agg_type == AggType.hgram:
                if isinstance(stat.bins, list):
                    agg_value = zip(stat.bins, [0] * (len(stat.bins) - 1))
                else:
                    agg_value = []
            else:
                msg = 'Aggregation for %s not implemented' % agg_type
                raise NotImplementedError(msg)

            results.append([stat.unistat_name, agg_value])

        return results


unistat_agent = UnistatAgent()
