# encoding: UTF-8\

import re

import enum

from appcore.struct import expect_type


class ValueType(str, enum.Enum):
    absolute = 'a'
    delta = 'd'


class AggType(str, enum.Enum):
    aver = 'v'
    max = 'x'
    min = 'n'
    summ = 'm'
    summone = 'e'
    trnsp = 't'
    counter = 'c'
    hgram = 'h'


class RollupType(str, enum.Enum):
    aver = 'v'
    max = 'x'
    min = 'n'
    summ = 'm'
    counter = 'c'
    hgram = 'h'


_UNISTAT_NAME_REGEX = re.compile(
    r'^[a-zA-Z0-9.\-/@_]+_[ad][vxnmetch]{2}[vxnmch]$',
)

_AGG_VALIDITY_MATRIX = {
    rollup: {
        agg: all((
            agg not in (AggType.hgram, AggType.counter),
            rollup not in (RollupType.hgram, RollupType.counter),
        ))
        for agg in AggType.__members__.values()
    }
    for rollup in RollupType.__members__.values()
}
_AGG_VALIDITY_MATRIX[RollupType.hgram][AggType.hgram] = True
_AGG_VALIDITY_MATRIX[RollupType.counter][AggType.counter] = True


class Stat(object):
    """
    Объекты класcа описывают unistat-метрики:

    name - имя метрики без sigopt.
    value_type - тип значения (абсолютный, дельта; транслируется в sigopt).
    hagg - метод агрегации сигнала на хосте (используется локально).
    gagg - метод агрегации сигнала в группе хостов (транслируется в sigopt).
    mgagg - метод агрегации сигнала в метагруппе (транслируется в sigopt).
    rollup - метод агрегации по времени (транслируется в sigopt).
    bins - настройка корзин для гистограмм (см. numpy.histogram).
    """

    def __init__(self, name, value_type, hagg, gagg, mgagg, rollup, bins=10):
        self.__name = expect_type(name, basestring)  # type: basestring
        self.__type = expect_type(value_type, ValueType)  # type: ValueType
        self.__hagg = expect_type(hagg, AggType)  # type: AggType
        self.__gagg = expect_type(gagg, AggType)  # type: AggType
        self.__mgagg = expect_type(mgagg, AggType)  # type: AggType
        self.__rollup = expect_type(rollup, RollupType)  # type: RollupType
        self.__bins = bins

        if not _AGG_VALIDITY_MATRIX[rollup][hagg]:
            raise ValueError('Host aggregation incompatible to rollup')

        if not _AGG_VALIDITY_MATRIX[rollup][gagg]:
            raise ValueError('Group aggregation incompatible to rollup')

        if not _AGG_VALIDITY_MATRIX[rollup][mgagg]:
            raise ValueError('Metagroup aggregation incompatible to rollup')

        unistat_sigopt = '%s%s%s%s' % (
            self.__type.value,
            self.__gagg.value,
            self.__mgagg.value,
            self.__rollup.value,
        )

        unistat_name = '%s_%s' % (
            self.__name,
            unistat_sigopt,
        )

        if not _UNISTAT_NAME_REGEX.match(unistat_name):
            raise ValueError('Invalid unistat name')

        self.__unistat_name = unistat_name

    @property
    def name(self):
        return self.__name

    @property
    def type(self):
        return self.__type

    @property
    def hagg(self):
        return self.__hagg

    @property
    def gagg(self):
        return self.__gagg

    @property
    def mgagg(self):
        return self.__mgagg

    @property
    def rollup(self):
        return self.__rollup

    @property
    def unistat_name(self):
        return self.__unistat_name

    @property
    def bins(self):
        return self.__bins

    @classmethod
    def summ(cls, name):
        return cls(
            name,
            ValueType.absolute,
            AggType.summ,
            AggType.summ,
            AggType.summ,
            RollupType.summ,
        )

    @classmethod
    def hgram(cls, name, bins=10):
        return cls(
            name,
            ValueType.absolute,
            AggType.hgram,
            AggType.hgram,
            AggType.hgram,
            RollupType.hgram,
            bins,
        )

    @classmethod
    def max(cls, name):
        return cls(
            name,
            ValueType.absolute,
            AggType.max,
            AggType.max,
            AggType.max,
            RollupType.max,
        )

    @classmethod
    def min(cls, name):
        return cls(
            name,
            ValueType.absolute,
            AggType.min,
            AggType.min,
            AggType.min,
            RollupType.min,
        )
