# encoding: UTF-8

import functools

import flask

from appcore.data.model import Pageable
from appcore.struct import coalesce
from appcore.web.exceptions import RenderError
from appcore.web.helpers import abort


def supports_pagination(
        func=None,
        default_size=None,
        offset_request_arg=None,
        size_request_arg=None,
):
    def pagination_wrapper(func):
        @functools.wraps(func)
        def paginated(*args, **kwargs):
            # Resolve arg names
            offset_arg = coalesce(
                offset_request_arg,
                flask.current_app.config.get('PAGINATION_OFFSET_REQUEST_ARG'),
                'offset',
            )
            size_arg = coalesce(
                size_request_arg,
                flask.current_app.config.get('PAGINATION_OFFSET_SIZE_ARG'),
                'size',
            )

            # Resolve offset
            offset = flask.request.args.get(offset_arg)
            if offset:
                try:
                    offset = int(offset)
                except ValueError:
                    abort(
                        400,
                        message='Invalid argument',
                        arg=offset_arg,
                    )
            else:
                offset = 0

            # Resolve size
            size = flask.request.args.get(size_arg)
            if size:
                try:
                    size = int(size)
                except ValueError:
                    abort(
                        400,
                        message='Invalid argument',
                        arg=size_arg,
                    )
            else:
                size = coalesce(
                    default_size,
                    flask.current_app.config.get('PAGINATION_DEFAULT_SIZE'),
                    20,
                )
                size = int(size)

            # Call decorated func with additional argument
            kwargs['pageable'] = Pageable(offset, size)
            return func(*args, **kwargs)

        # Store pagination settings for doc generation
        paginated.__paination__ = dict(
            default_size=default_size,
            offset_request_arg=offset_request_arg,
            size_request_arg=size_request_arg,
        )
        return paginated

    if func is None:
        return pagination_wrapper
    else:
        return pagination_wrapper(func)


def parse_body(schema_cls, *schema_args, **schema_kwargs):
    def parse_body_wrapper(func):
        @functools.wraps(func)
        def body_parser(*args, **kwargs):
            schema = schema_cls(*schema_args, **schema_kwargs)

            body, errors = schema.load(flask.request.json or {})

            if errors:
                abort(
                    422,
                    message='Invalid request body',
                    errors=errors,
                )

            kwargs['body'] = body
            return func(*args, **kwargs)

        body_parser.__parse_body__ = dict(
            shema_cls=schema_cls,
            schema_args=schema_args,
            schema_kwargs=schema_kwargs,
        )
        return body_parser

    return parse_body_wrapper


def _call_view(func, args, kwargs):
    data = func(*args, **kwargs)

    if isinstance(data, tuple):
        return data
    else:
        return data, None


def render(schema_cls, status_code=None, **schema_kwargs):
    def render_wrapper(func):
        @functools.wraps(func)
        def renderer(*args, **kwargs):
            schema = schema_cls(**schema_kwargs)

            data, explicit_status_code = _call_view(func, args, kwargs)

            data, errors = schema.dump(data)

            if errors:
                raise RenderError('Failed to render', errors)

            response = flask.jsonify(data)
            response.status_code = (
                    explicit_status_code or
                    status_code or
                    response.status_code
            )

            return response

        renderer.__render__ = dict(
            shema_cls=schema_cls,
            status_code=status_code,
            schema_kwargs=schema_kwargs,
        )
        return renderer

    return render_wrapper


def render_page(schema_cls, status_code=None, *schema_args, **schema_kwargs):
    schema_kwargs['many'] = True

    def render_page_wrapper(func):
        @functools.wraps(func)
        def page_renderer(*args, **kwargs):
            schema = schema_cls(*schema_args, **schema_kwargs)

            page, explicit_status_code = _call_view(func, args, kwargs)

            items, errors = schema.dump(page.items)

            if errors:
                raise RenderError('Failed to render', errors)

            response = flask.jsonify(
                dict(
                    items=items,
                    pagination=dict(
                        offset=page.offset,
                        size=page.size,
                        total=page.total,
                    ),
                )
            )
            response.status_code = (
                    explicit_status_code or
                    status_code or
                    response.status_code
            )

            return response

        page_renderer.__render__ = dict(
            shema_cls=schema_cls,
            status_code=status_code,
            schema_args=schema_args,
            schema_kwargs=schema_kwargs,
        )
        return page_renderer

    return render_page_wrapper


def punycode_domain_name(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        if 'domain_name' in kwargs:
            kwargs['domain_name'] = kwargs['domain_name'].encode('idna')
        return func(*args, **kwargs)

    return wrapper
