# encoding: UTF-8

import flask
import werkzeug.exceptions
import werkzeug.http

from appcore.plugin import PluginBase


class JsonWebErrorMixin(Exception):
    def _get_status_code(self):
        raise NotImplementedError

    def _get_headers(self):
        return {}

    def _get_details(self):
        return None

    def _get_body(self):
        body = dict(message=self.message)

        details = self._get_details()
        if details:
            body.update(details=details)

        return body

    def get_response(self):
        status_code = self._get_status_code()
        headers = self._get_headers()
        body = self._get_body()

        response = flask.jsonify(body)
        response.status_code = status_code
        response.headers.extend(headers)
        return response


class JsonWebErrorsPlugin(PluginBase):
    def _do_init_app(self):
        self.app.config['TRAP_HTTP_EXCEPTIONS'] = False
        self.app.config['TRAP_BAD_REQUEST_ERRORS'] = False

        for status in werkzeug.exceptions.default_exceptions:
            message = werkzeug.http.HTTP_STATUS_CODES[status]
            handler = self._make_handler(status, message)
            self.app.register_error_handler(status, handler)

        self.app.register_error_handler(
            JsonWebErrorMixin,
            self._json_web_error_handler,
        )

    # noinspection PyMethodMayBeStatic
    def _make_handler(self, status, message):
        def handler(_):
            return flask.make_response(
                flask.jsonify(
                    message=message,
                ),
                status,
            )

        return handler

    # noinspection PyMethodMayBeStatic
    def _json_web_error_handler(
            self,
            error,  # type: JsonWebErrorMixin
    ):
        return error.get_response()
