# encoding: UTF-8

import uuid

import flask
import ylog

from appcore.plugin import PluginBase


def generate_request_id():
    return uuid.uuid4().hex


def gunicorn_get_request_id(req):
    for k, v in req.headers:
        if k == RequestIDPlugin.REQUEST_ID_HEADER.upper():
            return v

    return generate_request_id()


def gunicorn_set_request_id(req_or_resp, request_id):
    header = (
        RequestIDPlugin.REQUEST_ID_HEADER.upper(),
        request_id,
    )
    if header not in req_or_resp.headers:
        req_or_resp.headers.append((
            RequestIDPlugin.REQUEST_ID_HEADER.upper(),
            request_id,
        ))

    ylog.context.put_to_context('request_id', request_id)


class RequestIDPlugin(PluginBase):
    REQUEST_ID_HEADER = 'X-Request-ID'
    REQUEST_ID_G = 'request_id'

    def _do_init_app(self):
        self.app.before_request(self._detect_request_id)
        self.app.after_request(self._copy_request_id_to_response)

    def _detect_request_id(self):
        request_id = flask.request.headers.get(self.REQUEST_ID_HEADER)
        if request_id is None:
            request_id = generate_request_id()

        setattr(flask.g, self.REQUEST_ID_G, request_id)
        ylog.context.put_to_context('request_id', request_id)

    def _copy_request_id_to_response(self, response):
        request_id = getattr(flask.g, self.REQUEST_ID_G)
        response.headers[self.REQUEST_ID_HEADER] = request_id
        return response


def get_request_id():
    return getattr(flask.g, RequestIDPlugin.REQUEST_ID_G, None)
