import time
from typing import Dict, Union
from urllib.parse import urljoin

from aiohttp import ClientTimeout, ClientResponseError
from aiohttp_retry import RetryClient, ExponentialRetry

from settings import config
from utils.stats import (
    http_client_request_status, http_client_response_time, http_client_response_status,
    http_client_method_response_time, http_client_request_count, Status,
    http_client_response_short_status, http_client_method_response_short_status
)
from utils.tvm import get_service_ticket


class BaseHttpClient:
    base_url: str = None
    oauth_token: str = None
    tvm_client_id: str = None
    raise_for_status: bool = True
    timeout: float = 20.0
    service: str = None  # для метрик, будет в названии сигнала

    def __init__(self):
        headers = {
            "X-Host": config.hostname,
            "User-Agent": config.useragent,
        }
        if self.oauth_token:
            headers['Authorization'] = f'OAuth {self.oauth_token}'

        retry_options = ExponentialRetry(
            attempts=3,
            statuses=set(range(400, 600)),
        )

        self._session = RetryClient(
            retry_options=retry_options,
            headers=headers,
            raise_for_status=self.raise_for_status,
            timeout=ClientTimeout(total=self.timeout),
        )

    async def request(
        self, method: str, path: str, headers: Dict = None, json_encode: bool = True, **kwargs
    ) :
        http_client_request_count.labels(self.service).inc()
        success = True
        start = time.monotonic()
        response = None
        url = self.create_url(path)
        if not headers:
            headers = {}
        try:
            if self.tvm_client_id:
                headers["X-Ya-Service-Ticket"] = await get_service_ticket(self.tvm_client_id)

            async with self._session._request(method=method, url=url, headers=headers, **kwargs) as response:
                return await response.json() if json_encode else await response.read()
        except ClientResponseError as e:
            # в случае http ошибки и включенного raise_for_status надо в метрики отдать код ответа
            response = e
            success = False
            raise e
        except Exception as e:
            success = False
            raise e
        finally:
            response_time = time.monotonic() - start
            response_status = response.status if response else 599
            self._response_save_metrics(method, response_time, response_status, success)


    async def get(self, *args, **kwargs) -> Union[Dict, str]:
        return await self.request("get", *args, **kwargs)

    async def post(self, *args, **kwargs) -> Union[Dict, str]:
        return await self.request("post", *args, **kwargs)

    def create_url(self, url):
        return urljoin(self.base_url.rstrip('/') + '/', url.lstrip('/')) if self.base_url else url

    def _response_save_metrics(self, method: str, response_time: float, status: int, success: bool):
        http_client_response_time.labels(self.service).observe(response_time)
        http_client_method_response_time.labels(self.service, method).observe(response_time)

        short_status = str(status // 100) + 'xx'
        http_client_response_status.labels(self.service, status).inc()
        http_client_response_short_status.labels(self.service, short_status).inc()
        http_client_method_response_short_status.labels(self.service, method, short_status).inc()

        http_client_request_status.labels(self.service, Status.success.value if success else Status.failed.value).inc()
