import re
import time
from abc import abstractmethod
from datetime import datetime
from typing import Dict

from loguru import logger

from settings import Settings
from settings.base import BaseTaskSettings
from utils.stats import tasks_counter, tasks_count_gauge, Status
from utils.utils import camel_to_snake


class BaseTask:
    def __init__(self, config: Settings):
        self._name = camel_to_snake(re.sub("Task$", "", self.__class__.__name__))
        self._locke_path = f'{config.yt.locke_path}/{config.env}/{self._name}'
        self.config: BaseTaskSettings = getattr(config, f'{self._name}')

    @abstractmethod
    async def _run(self):
        ...

    async def run(self):
        logger.info(f"{self._name} task is starting")
        if not self.config.enabled:
            logger.info(f"{self._name} task is disabled")
            return
        success = True
        self.start_time = time.monotonic()
        tasks_count_gauge.labels(self._name).inc()
        try:
            await self._run()
        except Exception as e:
            logger.exception(f"task execution error: {e}")
            success = False
        finally:
            logger.info(f"{self._name} task is finished")
            tasks_count_gauge.labels(self._name).dec()
            self._task_save_metrics(success)

    def get_scheduler_settings(self) -> Dict:
        return {
            "hours": self.config.scheduler.hours,
            "trigger": self.config.scheduler.trigger,
            "id": self._name,
            "next_run_time": datetime.now() if self.config.scheduler.start_now else None,
        }

    def _task_save_metrics(self, success: bool):
        tasks_counter.labels(self._name, Status.success.value if success else Status.failed.value).inc()
