import asyncio
import concurrent.futures
import re
import time
from datetime import datetime, timedelta
from typing import Set, Optional, List, Dict

from aiohttp import ClientResponseError
from loguru import logger
from yql.api.v1.client import YqlClient

from clients import DNSAPIHttpClient, CatalogHttpClient
from clients.yt import set_lock
from settings import config, Settings
from settings.base import CheckCurrentYearSettings
from tasks.base import BaseTask
from utils.stats import domains_gauge, DomainStatus, tasks_time
from utils.utils import (
    test_domains,
    gendarme_on_date_query,
    grouper,
    has_cyrillic,
    to_punycode,
    gendarme_interval_query,
    DATE_FORMAT,
    ya_dns,
)

nservers_pattern = re.compile('.*\s*\d+\sin\sns\s.*')
nservers_split_pattern = re.compile('in\s*ns\s*')

google_dns_list = ['2001:4860:4860::8888', '2001:4860:4860::8844']


class CheckCurrentYearTask(BaseTask):
    def __init__(self, config: Settings):
        super().__init__(config)
        self.config: CheckCurrentYearSettings
        self.catalog_client = CatalogHttpClient()
        self.dns_api_client = DNSAPIHttpClient()
        self.yql_client = YqlClient()
        self.date = self.config.date or datetime.now().strftime('%Y')
        self.days = self.config.days
        self.dig_timeout = self.config.dig_timeout
        self.dig_tries = self.config.dig_tries
        self.created_domains = []
        self.processed_domains_count = 0

    @set_lock
    async def _run(self):
        domains = await self.domains_on_date()
        await self.run_once(domains)

    async def run_once(self, domains: Set[str]):
        self.created_domains = []
        self.processed_domains_count = 0

        if not domains:
            logger.info('no new delegated domains from gendarme')
            return

        logger.debug(f'found {len(domains)} domains from gendarme')
        await self.domains_processing(domains)

    async def domains_processing(self, domains: Set[str]):
        for chunk in grouper(self.config.chunk_size, domains, fillvalue=""):
            logger.debug(f'domains chunk from gendarme: {chunk}')
            await self.group_domains_processing(chunk)

    async def group_domains_processing(self, chunk: List):
        try:
            exist_domains = await self.catalog_client.search_domains(chunk)
        except Exception as e:
            logger.error(f'error from catalog: {e}')
            return

        logger.debug(f'{len(exist_domains)} domains exist in catalog: {exist_domains}')
        # фильтруем значения по признаку owned
        owned_domains = self.get_owned_domains(exist_domains)
        logger.info(f'{len(owned_domains)} verified domains: {owned_domains}')
        if not owned_domains:
            self.processed_domains_count += len(chunk)
            return
        # проверяем ns подтвержденных доменов
        delegated_domains = await self.get_delegated_domains(owned_domains)

        # проверяем какие из делегированных доменов нет в dns-hosing
        not_exist_domains = await self.exists_domains(owned_domains, delegated_domains)
        logger.debug(f'{len(not_exist_domains)} domains need added to DNS hosting: {not_exist_domains}')

        # добавляем несуществующие домены в dns-hosting
        enabled_domains = await self.bulk_enables_domains(not_exist_domains)
        if enabled_domains:
            logger.info(f'{len(enabled_domains)} domains added to DNS hosting: {enabled_domains}')

        self.created_domains.extend(enabled_domains)
        self.processed_domains_count += len(chunk)

        logger.info(f'processed domains: {self.processed_domains_count}')
        logger.info(f'{len(self.created_domains)} domains added: {self.created_domains}')
        self._save_task_metrics(exist_domains, delegated_domains, not_exist_domains )

    async def bulk_enables_domains(self, not_exist_domains) -> List:
        results = await asyncio.gather(
            *[self.enable_domain(domain["domain_name"], domain["org_id"]) for domain in not_exist_domains],
            return_exceptions=True,
        )
        exceptions = list(filter(lambda r: isinstance(r, Exception), results))
        if exceptions:
            logger.warning(f'exceptions from dns-api: {exceptions}')

        return list(
            filter(lambda r: r is not None and not isinstance(r, Exception), results)
        )

    async def get_delegated_domains(self, owned_domains: Dict) -> List:
        results = await asyncio.gather(
            *[self.check_delegation(domain) for domain in owned_domains],
            return_exceptions=True,
        )

        return list(
            filter(lambda r: r is not None and not isinstance(r, Exception), results)
        )

    async def exists_domains(self, owned_domains: Dict, delegated_domains: List) -> List:
        results = await asyncio.gather(
            *[self.domain_exist_in_dns(domain, owned_domains[domain]) for domain in delegated_domains],
            return_exceptions=True,
        )

        exceptions = list(filter(lambda r: isinstance(r, Exception), results))
        if exceptions:
            logger.warning(f'exceptions from dns-api: {exceptions}')

        not_exist_domains = list(
            filter(lambda r: r is not None and not isinstance(r, Exception), results)
        )
        return not_exist_domains

    async def domain_exist_in_dns(self, domain_name: str, org: Dict[str, str]) -> Optional[Dict]:
        org_id = str(org.get('org_id'))

        try:
            resp = await self.dns_api_client.get_domain(domain_name, org_id)
            logger.debug(f'{domain_name} response from dns-api: {resp}')
        except ClientResponseError as e:
            if e.status == 404:
                logger.info(f'{domain_name} not exist in dns-api')
                return {
                    "domain_name": domain_name,
                    "org_id": org_id,
                }

        return None

    async def enable_domain(self, domain_name: str, org_id: str) -> Optional[Dict]:
        try:
            resp = await self.dns_api_client.enable_domain(domain_name, org_id)
            logger.debug(f'{domain_name} response from dns-api: {resp}')
        except ClientResponseError as e:
            logger.error(f'{domain_name} http client error from dns-api: {e.status} {e.message} ')
            return None

        return resp

    async def domains_on_date(self) -> Set[str]:
        if config.env != 'production':
            # костыль для test/dev, так как у жандарма нет логов кроме продовых
            return test_domains

        query = self.build_gendarme_query()

        loop = asyncio.get_running_loop()
        with concurrent.futures.ThreadPoolExecutor() as pool:
            return await loop.run_in_executor(pool, self.get_domains_from_gendarme, query)

    def build_gendarme_query(self):
        now = datetime.now()
        if self.days:
            return gendarme_interval_query(
                date_from=(now - timedelta(days=self.days)).strftime(DATE_FORMAT), date_to=now.strftime(DATE_FORMAT)
            )

        return gendarme_on_date_query(self.date)

    def get_domains_from_gendarme(self, query: str) -> Set[str]:
        domains = set()

        request = self.yql_client.query(query, syntax_version=1)
        request.run()
        result = request.get_results()

        for table in result:
            table.fetch_full_data()
            domains = [str(row[0]) for row in table.rows]

        return domains

    async def check_delegation(self, domain: str) -> Optional[str]:
        # yaconnect.com делегированы на ns3, ns4.yandex.net - нет смысла проверять
        # mail.narod.ru - на ucoz
        if 'yaconnect.com' in domain or 'mail.narod.ru' in domain:
            return None

        # в dns-api надо передавать домен кириллицей, а во whois надо idna
        if has_cyrillic(domain):
            domain_name = to_punycode(domain)
        else:
            domain_name = domain

        whois_cmd = f'dig +trace +time={self.dig_timeout} +tries={self.dig_tries} {domain_name}'
        proc = await asyncio.create_subprocess_shell(
            whois_cmd, stdout=asyncio.subprocess.PIPE, stderr=asyncio.subprocess.PIPE
        )

        stdout, stderr = await proc.communicate()
        if stderr:
            logger.warning(f'{domain}: {stderr}')

        # вполне может быть что сфейлился только ответ одного сервера, при этом другие ответили нормально
        # поэтому выходим только если пустой stdout, а не по наличию stderr
        if not stdout:
            return None

        stdout = str(stdout.decode('utf-8')).lower()
        # находим регуляркой все dns сервера в ответе
        all_nservers: List[str] = nservers_pattern.findall(stdout)
        # находим dns сервера на которые делегирована этот домен, а не домны 1го уровня
        nameservers = [ns.strip('\n') for ns in all_nservers if ns.startswith(domain_name)]

        # получаем set из только имен dns серверов на которые делегирован домен
        nameservers = set(map(lambda n: re.split(nservers_split_pattern, n)[-1], nameservers))
        logger.debug(f'{domain} has a nameservers: {nameservers}')
        # if not nameservers:
        #     logger.debug("{domain} stdout: {std}".format(domain=domain, std=stdout.replace('\n', '')))
        # проверяем что только наши dns сервера есть в списке на которые делегирован домен
        if ya_dns == nameservers:
            logger.info(f'{domain} is delegated to Yandex nameservers')
            return domain

        return None

    def get_owned_domains(self, domains: Dict) -> Dict:
        owned_domains = {}
        for domain_name, org_list in domains.items():
            owned_org = self.filter_owned_organization(org_list)
            if not owned_org:
                continue
            owned_domains[domain_name] = owned_org
        return owned_domains

    @staticmethod
    def filter_owned_organization(org_list: List[Dict[str, str]]) -> Optional[Dict[str, str]]:
        """
        Находим в списке только ту организацию которая является подтвержденной.
        У домена должна быть только одна организация в owned = True.
        Если подтвержденных будет несколько, то возвращать должны первую - см. https://st.yandex-team.ru/DNSHOSTING-31
        """
        for org in org_list:
            if org.get('owned', False) and org.get('org_id'):
                return org
        return None

    def _save_task_metrics(self, exist_domains, delegated_domains, not_exist_domains):
        domains_gauge.labels(self._name, DomainStatus.exist.value).observe(len(exist_domains))
        domains_gauge.labels(self._name, DomainStatus.delegated.value).observe(len(delegated_domains))
        domains_gauge.labels(self._name, DomainStatus.need_created.value).observe(len(not_exist_domains))

        domains_gauge.labels(self._name, DomainStatus.processed.value).observe(self.processed_domains_count)
        domains_gauge.labels(self._name, DomainStatus.created.value).observe(len(self.created_domains))
        execution_time = time.monotonic() - self.start_time
        tasks_time.labels(self._name).observe(execution_time)
