from loguru import logger

from clients.yt import yt_client
from settings import config
from settings.base import CheckLatestYearsSettings, Settings
from tasks import CheckCurrentYearTask
from utils.utils import gendarme_on_date_query


class CheckLatestYearsTask(CheckCurrentYearTask):
    def __init__(self, config: Settings):
        super().__init__(config)
        self.config: CheckLatestYearsSettings
        # таймаут на лок ставим периодичность выполнения таска * 3, чтобы не слетел если вдруг таска будет выполняться дольше
        # потом стоит это переделать вместе с шедулером
        self.lock_timeout = int(self.config.scheduler.hours * 3 * 3600 * 1000)  # таймаут в msec ставится
        self.domains_offset = self.config.domains_offset  # по сколько доменов выгребаем за одну итерацию
        self.domain_count = 4742613  # доменов с 2018 по 2022
        self.current_limit = 0

    async def _run(self):
        await self.limit_selection()
        domains = await self.domains_on_date()
        await self.run_once(domains)
        # лок не удаляем после работы, так как ставим таймаут на их удаление, чтобы воркеры не начинали обрабатывать заново

    async def limit_selection(self):
        """Находит limit для выборки из логов по данным из locke"""
        # если не создана нода на джобу - то создаем и берем первую часть доменов
        if (
            await yt_client.create_locke_if_not_exist(path=self._locke_path, recursive=True)
            and await self.set_lock_on_limit()
        ):
            return

        # получаем текущий список обрабатываемых кусков - имя ноды == limit_to
        tree = await yt_client.get(path=self._locke_path)
        blocked_limits = sorted(list(map(lambda x: int(x), tree.keys())))

        # находим наибольший
        logger.info(f'all blocked limits: {blocked_limits}')
        last_limit = blocked_limits[-1] if blocked_limits else 0
        self.current_limit = last_limit
        while self.current_limit < self.domain_count:
            if await self.set_lock_on_limit():
                return
            # если вдруг уже лок успели занять, то берем следующую часть доменов
            self.current_limit += self.domains_offset

        # если все прошли домены - начинаем заново
        self.current_limit = 0
        for limit in blocked_limits:
            await yt_client.remove(f'{self._locke_path}/{limit}')
            logger.info(f'clear lock: {limit}')
        await self.set_lock_on_limit()

    async def set_lock_on_limit(self) -> bool:
        if await self.create_node(f'{self._locke_path}/{self.current_limit + self.domains_offset}'):
            logger.info(f'select start limit: {self.current_limit}')
            return True
        return False

    async def create_node(self, path: str) -> bool:
        attributes = {
            "expiration_timeout": self.lock_timeout,
            "host": config.hostname,
        }
        try:
            await yt_client.create(path=path, type='map_node', attributes=attributes)
        except Exception as e:
            logger.info(f'can not create node {path}: {e}')
            return False
        return True

    def build_gendarme_query(self):
        return gendarme_on_date_query(
            self.date, self.current_limit, self.domains_offset, not_like=True, yesterday_log=True
        )
