import logging
import time
from sys import stderr

from loguru import logger, _better_exceptions
from ujson import dumps

from settings import config


class InterceptHandler(logging.Handler):
    # https://loguru.readthedocs.io/en/stable/overview.html?highlight=InterceptHandler#entirely-compatible-with-standard-logging
    def emit(self, record):
        try:
            level = logger.level(record.levelname).name
        except ValueError:
            level = record.levelno

        frame, depth = logging.currentframe(), 2
        while frame.f_code.co_filename == logging.__file__:
            frame = frame.f_back
            depth += 1

        logger.opt(depth=depth, exception=record.exc_info).log(level, record.getMessage())


def deploy_formatter(record):
    record["extra"]["serialized"] = dumps(
        {
            "@timestamp": record["time"].isoformat(),
            "message": record["message"],
            "stackTrace": "".join(
                _better_exceptions.ExceptionFormatter().format_exception(
                    record["exception"].type,
                    record["exception"].value,
                    record["exception"].traceback
                )
            )
            if record["exception"]
            else None,
            "levelStr": record["level"].name,
            "level": record["level"].no,
            "loggerName": f"{record['name']}:{record['function']}",
            # extra
            "line": record["line"],
            "threadName": record["file"].path,
        }
    )
    return "{extra[serialized]}\n"

class TskvFormatter(logging.Formatter):
    FIELDS = [
        ('levelname', 'levelStr'),
        ('asctime', '@timestamp'),
        ('name', 'loggerName'),
        ('funcName', 'func_name'),
        ('pathname', 'filename'),
        ('module', 'module'),
        ('process', 'pid'),
        ('message', 'message'),
    ]

    TSKV_PREFIX = 'tskv'

    def quote(self, value):
        return (str(value).
                replace('=', '\\=').
                encode('unicode_escape').
                decode('ascii'))

    def formatTime(self, record, datefmt=None):
        gt = time.gmtime(record.created)
        return time.strftime('%Y-%m-%dT%H:%M:%S.%%03dZ', gt) % (record.msecs,)

    def format(self, record):
        record.message = record.getMessage()
        record.asctime = self.formatTime(record)

        parts = [self.TSKV_PREFIX]
        for field, key in self.FIELDS:
            if hasattr(record, field):
                value = getattr(record, field)
                if key == 'filename':
                    value += ':' + str(getattr(record, 'lineno'))
                parts.append(self.quote(key) + '=' + self.quote(value))


        if record.exc_info and not record.exc_text:
            record.exc_text = self.formatException(record.exc_info)
        if record.exc_text:
            parts.append(self.quote('stackTrace') + '=' + self.quote(record.exc_text))

        return '\t'.join(parts)

def initialize_logging():
    logger.remove()
    if config.logging.stdout_enabled:
        format = {"format": deploy_formatter} if config.logging.stdout_json_formatted else {}
        logger.add(stderr, enqueue=True, **format)
    if config.logging.file_enabled:
        logger.add(
            config.logging.file_path,
            enqueue=True,
            rotation=config.logging.file_rotation_rule,
            retention=config.logging.file_retention_count,
            compression="gz",
        )
    logging.basicConfig(handlers=[InterceptHandler()], level=0)
