import asyncio
import contextvars
import re
from datetime import datetime, timedelta
from functools import partial, wraps
from itertools import zip_longest
from typing import Any, Iterable


DATE_FORMAT = "%Y-%m-%d"

test_domains = [
    'selo.connect-test.tk',
    'yaconnect.com',
    'sup.auto.connect-tk.tk',
    'du.connect-test.tk',
    'kasha.connect-test.tk',
    'dzen1.auto.connect-test.tk',
    'sukin.auto.connect-test.tk',
    'роллсистем.рф',
]

ya_dns = {
    'dns1.yandex.net.',
    'dns2.yandex.net.',
}


base_gendarme_query = "pragma yt.InferSchema; use hahn;\nSELECT distinct(`domain`) from `//home/mail-pdd/gendarme"


def gendarme_on_date_query(
    date: str, limit_from: int = None, limit_to: int = None, not_like: bool = False, yesterday_log: bool = False
) -> str:
    log_date = datetime.now() if not yesterday_log else datetime.now() - timedelta(days=1)

    query = (
        f"{base_gendarme_query}/{log_date.strftime(DATE_FORMAT)}` "
        f"WHERE `created_at` {'NOT' if not_like else ''} LIKE '{date}%' ORDER BY domain DESC"
    )
    if limit_from is not None:
        query += f" LIMIT {limit_from}"
    if limit_to is not None:
        query += f", {limit_to}"
    query += ";"
    return query


def gendarme_interval_query(date_from: str, date_to: str) -> str:
    return (
        f"{base_gendarme_query}/{datetime.now().strftime(DATE_FORMAT)}` "
        f"WHERE `created_at` >= '{date_from}T00:00:00' and `created_at` <= '{date_to}T23:59:59' ORDER BY domain DESC;"
    )


def grouper(n: int, iterable: Iterable, fillvalue: Any = None) -> Iterable:
    """grouper(3, 'ABCDEFG', 'x') --> ABC DEF Gxx"""
    args = [iter(iterable)] * n
    return zip_longest(fillvalue=fillvalue, *args)


def has_cyrillic(text: str) -> bool:
    return bool(re.search('[а-яА-Я]', text))


def to_punycode(string: str):
    return string.encode("idna").decode("utf-8")


def sync_to_async(func):
    @wraps(func)
    async def wrapper(*args, loop=None, executor=None, **kwargs):
        if loop is None:
            loop = asyncio.get_event_loop()

        context = contextvars.copy_context()
        p_func = partial(context.run, func, *args, **kwargs)
        return await loop.run_in_executor(executor, p_func)

    return wrapper


camel_pattern = re.compile(r"(?<!^)(?=[A-Z])")


def camel_to_snake(s: str):
    return camel_pattern.sub("_", s).lower()
