# encoding: UTF-8

import logging
import os

import flask
from ws_properties.environ.properties import DictPropertySource

from appcore.injection import register_injection
from dns_hosting.config.cli import configure_cli
from dns_hosting.config.dao import configure_dao
from dns_hosting.config.database import configure_database
from dns_hosting.config.dnsmaster import configure_dnsmaster
from dns_hosting.config.models import configure_models
from dns_hosting.config.security import configure_security
from dns_hosting.config.unistat import configure_unistat
from dns_hosting.config.web import configure_web


def create_app(environment):
    """
    Создает и конфигурирует экзепляр приложения.

    :param environment: Словарь с настройками приложения.
    :return: Экземпляр приложения.
    """

    assets_root = os.path.join(os.path.dirname(__file__), 'assets')
    assets_property_source = DictPropertySource(dict(assets_root=assets_root))
    environment.property_sources.insert(0, assets_property_source)

    app = flask.Flask(__name__)

    # Форсим обычный логер для Flask приложения, чтобы на него распространялись
    # настройки root-логгера
    app._logger = logging.getLogger(app.import_name)

    # делаем настройки окружения доступным для инъекций
    register_injection(app, 'environment', environment)

    # настройки unistat
    configure_unistat(app, environment)
    # Настройка БД
    configure_database(app, environment)
    # Настройка моделей
    configure_models(app, environment)
    # Настройка DAO (Data Access Object)
    configure_dao(app, environment)
    # Настройки консольных комманд
    configure_cli(app, environment)
    # Настройки веба
    configure_web(app, environment)
    # Настройки безопасности
    configure_security(app, environment)

    # Настройка DNS Master Server
    configure_dnsmaster(app, environment)

    return app
