# encoding: UTF-8

import time

import click
import sqlalchemy.engine as engine
from werkzeug.utils import cached_property
from ws_properties.environ.environment import Environment
from ws_properties.environ.mapper import ObjectMapper, ValueMapper
from ws_properties.utils.logs import get_logger_for_instance

from appcore.injection import Injected
from dns_hosting.services.migrator.service import MigratorService
from dns_hosting.utils.cli import ApplicationCommand

MIGRATE_PROPERTIES = ObjectMapper(
    writers=ValueMapper(int),
    workers=ValueMapper(int),
    write_batch_size=ValueMapper(int),
    read_batch_size=ValueMapper(int),
    queue_size=ValueMapper(int),
    interval=ValueMapper(float),
    log_interval=ValueMapper(float),
)


class PddMigrateCommand(ApplicationCommand):
    environment = Injected('environment')  # type: Environment
    pdd_engine = Injected('pdd_engine')  # type: engine.Engine
    slave_engine = Injected('slave_engine')  # type: engine.Engine
    master_engine = Injected('master_engine')  # type: engine.Engine

    def __init__(self, *args, **kwargs):
        super(PddMigrateCommand, self).__init__(*args, **kwargs)
        self._logger = get_logger_for_instance(self)

    @cached_property
    def properties(self):
        return MIGRATE_PROPERTIES.map(
            self.environment,
            'application.pdd.migrate',
        )

    def _iteration(self, migrator_service):
        migrator_service.migrate()

    def callback(self):
        migrator_service = MigratorService(
            self.properties['workers'],
            self.properties['writers'],
            self.properties['read_batch_size'],
            self.properties['write_batch_size'],
            self.properties['queue_size'],
            self.properties['log_interval'],
        )

        last_ts = time.time() - self.properties['interval']
        try:
            while True:
                if time.time() - last_ts >= self.properties['interval']:
                    last_ts = time.time()
                    self._iteration(migrator_service)
                time.sleep(0.1)
        except KeyboardInterrupt:
            self._logger.info('Terminating...')


@click.group()
def pdd():
    pass


pdd.add_command(PddMigrateCommand('migrate'))
