# encoding: UTF-8

import sys
import time

import click
import gunicorn.arbiter
import gunicorn.config
import ylog
from gunicorn.http.wsgi import Response
from ws_properties.environ.mapper import ObjectMapper
from ws_properties.environ.mapper import ValueMapper

from appcore.injection import inject
from appcore.unistat.agent import unistat_agent
from appcore.web.request_id import gunicorn_get_request_id
from appcore.web.request_id import gunicorn_set_request_id
from dns_hosting.stats import Stats

GUNICORN_PROPERTIES = ObjectMapper(
    bind=ValueMapper(str, default='[::]:80'),
    backlog=ValueMapper(int, default=2048),
    reuse_port=ValueMapper(bool, default=True),
    keepalive=ValueMapper(int, default=2),
    timeout=ValueMapper(int, default=30),
    graceful_timeout=ValueMapper(int, default=30),
    workers=ValueMapper(int, 1),
    pidfile=ValueMapper(str, default=None),
    worker_tmp_dir=ValueMapper(str, default=None),
    logconfig=ValueMapper(str, default='${assets_root}/gunilog.cfg'),
)


def gunicorn_pre_request(worker, req):
    request_id = gunicorn_get_request_id(req)
    gunicorn_set_request_id(req, request_id)

    req.started_at = time.time()

    unistat_agent.put(Stats.requests_count, 1)


def gunicorn_post_request(worker, req, environ, resp):
    if not resp.headers_sent:
        request_id = gunicorn_get_request_id(req)
        gunicorn_set_request_id(resp, request_id)

    started_at = req.started_at
    finished_at = time.time()
    request_time = finished_at - req.started_at

    ctx = ylog.context.log_context(
        request_started_at=started_at,
        request_finished_at=finished_at,
        request_time=request_time,
    )
    with ctx:
        worker.log.info('Request successfully handled')

    stat_values = [
        (Stats.requests_time, request_time),
    ]

    try:
        if resp.status_code == 500:
            stat_values.append((Stats.http_500_count, 1))

        elif resp.status_code == 499:
            stat_values.append((Stats.http_499_count, 1))
    except AttributeError:
        worker.log.exception('This happens in case of worker timeout.')

    unistat_agent.put_all(stat_values)


@click.command()
@click.pass_obj
def run_server(app):
    environment = inject('environment', app)
    properties = GUNICORN_PROPERTIES.map(environment, 'gunicorn')

    config = gunicorn.config.Config()

    for k, v in properties.items():
        config.set(k, v)

    config.set('daemon', False)
    config.set('preload_app', False)
    config.set('pre_request', gunicorn_pre_request)
    config.set('post_request', gunicorn_post_request)

    class App(object):
        cfg = config

        def wsgi(self):
            return app

    try:
        gunicorn.arbiter.Arbiter(App()).run()
    except RuntimeError as e:
        click.echo("\nError: %s\n" % e, err=True)
        sys.stderr.flush()
        sys.exit(1)
