# encoding: UTF-8

import sqlalchemy as sa
from ws_properties.environ.mapper import ObjectMapper, ValueMapper

from appcore.injection import register_injection
from dns_hosting.utils.postgres10 import make_postgres10_url

DB_PROPERTIES = ObjectMapper(
    url=ValueMapper(str),
    timeout=ValueMapper(int, default=4),
    echo=ValueMapper(bool, default=False),
    server_side_cursors=ValueMapper(bool, default=False),
    pool=ObjectMapper(
        size_min=ValueMapper(int, default=1),
        size_max=ValueMapper(int, default=5),
        recycle=ValueMapper(float, default=30),
        timeout=ValueMapper(float, default=30),
        echo=ValueMapper(bool, default=False),
    ),
)


def create_engine(environment, name, **kwargs):
    opts = DB_PROPERTIES.map(environment, 'application.database.' + name)
    datacenter = environment.get_property('instance.datacenter')

    return sa.create_engine(
        make_postgres10_url(opts['url'], datacenter),
        connect_args=dict(connect_timeout=opts['timeout']),
        server_side_cursors=opts['server_side_cursors'],
        echo=opts['echo'],
        echo_pool=opts['pool']['echo'],
        pool_size=opts['pool']['size_min'],
        max_overflow=opts['pool']['size_max'] - opts['pool']['size_min'],
        pool_recycle=opts['pool']['recycle'],
        pool_timeout=opts['pool']['timeout'],
        pool_pre_ping=True,
        **kwargs
    )


def configure_database(app, environment):
    if 'db-master' in environment.profiles:
        master_engine = create_engine(
            environment,
            'master',
            client_encoding='utf8',
        )
        register_injection(app, 'master_engine', master_engine)

    if 'db-slave' in environment.profiles:
        slave_engine = create_engine(
            environment,
            'slave',
            client_encoding='utf8',
            pool_reset_on_return=None,
        )
        register_injection(app, 'slave_engine', slave_engine)

    if 'db-pdd' in environment.profiles:
        pdd_engine = create_engine(
            environment,
            'pdd',
        )
        register_injection(app, 'pdd_engine', pdd_engine)
