# encoding: UTF-8

import logging

import ticket_parser2.api.v1 as tp2
import tvm2
from ws_properties.environ.mapper import ObjectMapper, ValueMapper, ListMapper

from appcore.injection import register_injection
from appcore.security.identifying import StandardIdentifyingStrategy
from appcore.security.manager import SecurityManager
from appcore.security.plugin import SecurityPlugin
from dns_hosting.services.auth.authenticator import Tvm2RequestsAuthenticator
from dns_hosting.services.auth.blackbox import JsonBlackbox
from dns_hosting.services.auth.directory import DirectoryClient, \
    MockDirectoryClient
from dns_hosting.services.auth.fouras import FourasClient, \
    MockFourasClient
from dns_hosting.services.auth.model import Scope
from dns_hosting.services.auth.providers.debug import DebugAuthProvider
from dns_hosting.services.auth.providers.host_specific import \
    HostSpecificAuthProvider
from dns_hosting.services.auth.providers.oauth import OAuthProvider
from dns_hosting.services.auth.providers.tvm2 import Tvm2AuthProvider

_log = logging.getLogger(__name__)

SECURITY_PROPERTIES = ObjectMapper(
    allowed_external_hosts=ListMapper(str, min_occurrence=0),
    allowed_internal_hosts=ListMapper(str, min_occurrence=0),
    tvm2=ObjectMapper(
        url=ValueMapper(str),
        client=ObjectMapper(
            id=ValueMapper(int),
            secret=ValueMapper(str),
        ),
        destinations=ListMapper(str),
        allowed_clients=ListMapper(
            ObjectMapper(
                id=ValueMapper(int),
                scopes=ListMapper(
                    ValueMapper(Scope, by_name=True),
                ),
            )
        ),
    ),
    blackbox=ObjectMapper(
        url=ValueMapper(str),
        client_id=ValueMapper(tp2.BlackboxClientId, by_name=True),
    ),
    directory=ObjectMapper(
        url=ValueMapper(str),
        client_id=ValueMapper(str),
        verify=ValueMapper(bool),
        ca=ValueMapper(str),
    ),
    fouras=ObjectMapper(
        url=ValueMapper(str),
        client_id=ValueMapper(str),
        verify=ValueMapper(bool),
        ca=ValueMapper(str),
    ),
)


def configure_security(app, environment):
    if 'web' not in environment.profiles:
        return

    SecurityPlugin(app=app)

    identifying_strategy = StandardIdentifyingStrategy(
        ignored_endpoints=[
            'health.ping',
            'health.unistat',
            'health.echo',
        ],
    )
    register_injection(app, 'identifying_strategy', identifying_strategy)

    security_manager = SecurityManager(identifying_strategy)
    register_injection(app, 'security_manager', security_manager)

    if 'web-security-mock' in environment.profiles:
        configure_security_mock(app, environment, security_manager)
    elif 'web-security-debug' in environment.profiles:
        configure_security_debug(app, environment, security_manager)
    else:
        configure_security_real(app, environment, security_manager)


def configure_security_real(app, environment, security_manager):
    security_properties = SECURITY_PROPERTIES.map(
        environment,
        'application.security',
    )

    client_scopes = {
        client['id']: client['scopes']
        for client in security_properties['tvm2']['allowed_clients']
    }

    tvm2_client = tvm2.TVM2(
        client_id=security_properties['tvm2']['client']['id'],
        secret=security_properties['tvm2']['client']['secret'],
        blackbox_client=security_properties['blackbox']['client_id'],
        allowed_clients=client_scopes.keys(),
        destinations=security_properties['tvm2']['destinations'],
        api_url=security_properties['tvm2']['url']
    )
    register_injection(app, 'tvm2_client', tvm2_client)

    blackbox_client = JsonBlackbox(
        tvm2_client=tvm2_client,
        **security_properties['blackbox']
    )
    register_injection(app, 'blackbox_client', blackbox_client)

    oauth_provider = HostSpecificAuthProvider(
        OAuthProvider(blackbox_client),
        security_properties['allowed_external_hosts'],
    )
    security_manager.auth_providers.append(oauth_provider)
    register_injection(app, 'oauth_provider', oauth_provider)

    tvm2_auth_provider = HostSpecificAuthProvider(
        Tvm2AuthProvider(tvm2_client, client_scopes),
        security_properties['allowed_internal_hosts'],
    )
    security_manager.auth_providers.append(tvm2_auth_provider)
    register_injection(app, 'tvm2_auth_provider', tvm2_auth_provider)

    directory_client = DirectoryClient(
        security_properties['directory']['url'],
        security_properties['directory']['verify'],
        security_properties['directory']['ca'],
        Tvm2RequestsAuthenticator(
            tvm2_client,
            security_properties['directory']['client_id'],
        )
    )
    register_injection(app, 'directory_client', directory_client)

    fouras_client = FourasClient(
        security_properties['fouras']['url'],
        security_properties['fouras']['verify'],
        security_properties['fouras']['ca'],
        Tvm2RequestsAuthenticator(
            tvm2_client,
            security_properties['fouras']['client_id'],
        )
    )
    register_injection(app, 'fouras_client', fouras_client)


def configure_security_debug(app, environment, security_manager):
    security_properties = SECURITY_PROPERTIES.map(
        environment,
        'application.security',
    )

    client_scopes = {
        client['id']: client['scopes']
        for client in security_properties['tvm2']['allowed_clients']
    }

    tvm2_client = tvm2.TVM2(
        client_id=security_properties['tvm2']['client']['id'],
        secret=security_properties['tvm2']['client']['secret'],
        blackbox_client=tp2.BlackboxClientId.Test,
        allowed_clients=client_scopes.keys(),
        destinations=security_properties['tvm2']['destinations'],
        api_url=security_properties['tvm2']['url']
    )
    register_injection(app, 'tvm2_client', tvm2_client)

    debug_auth_provider = DebugAuthProvider()
    security_manager.auth_providers.append(debug_auth_provider)

    directory_client = DirectoryClient(
        security_properties['directory']['url'],
        security_properties['directory']['verify'],
        security_properties['directory']['ca'],
        Tvm2RequestsAuthenticator(
            tvm2_client,
            security_properties['directory']['client_id'],
        )
    )
    register_injection(app, 'directory_client', directory_client)

    fouras_client = FourasClient(
        security_properties['fouras']['url'],
        security_properties['fouras']['verify'],
        security_properties['fouras']['ca'],
        Tvm2RequestsAuthenticator(
            tvm2_client,
            security_properties['fouras']['client_id'],
        )
    )
    register_injection(app, 'fouras_client', fouras_client)


def configure_security_mock(app, environment, security_manager):
    debug_auth_provider = DebugAuthProvider()
    security_manager.auth_providers.append(debug_auth_provider)

    directory_client = MockDirectoryClient()
    register_injection(app, 'directory_client', directory_client)

    fouras_client = MockFourasClient()
    register_injection(app, 'fouras_client', fouras_client)

