# encoding: UTF-8

from __future__ import absolute_import

import blackbox as bb
import tvm2


class JsonBlackbox(bb.JsonBlackbox):
    """
    Мы хотим ходить в blackbox с TVM тикетом, а
    стандартная либа этого не поддерживает, поэтому
    мы добавляем заголовок в этом классе, переопределив метод
    _blackbox_json_call.

    Так же в случае ошибок мы хотим получать исключение, поэтому в
    _blackbox_json_call добавлен код для его генерации.
    """

    def __init__(self, url, client_id, tvm2_client, *args, **kwargs):
        super(JsonBlackbox, self).__init__(url, *args, **kwargs)
        self.client_id = client_id
        self.tvm2 = tvm2_client  # type: tvm2.TVM2

    def _get_service_ticket(self):
        tickets = self.tvm2.get_service_tickets(self.client_id.value)
        ticket = tickets.get(self.client_id.value)

        if not ticket:
            raise bb.BlackboxError('Can not obtain service ticket')

        return ticket

    def _blackbox_json_call(self, method, **kwargs):
        kwargs = dict(kwargs)
        headers = kwargs.setdefault('headers', {})
        headers.update({
            'X-Ya-Service-Ticket': self._get_service_ticket(),
        })

        response = super(JsonBlackbox, self)._blackbox_json_call(
            method,
            **kwargs
        )

        if 'error' in response and response['error'] != 'OK':
            raise bb.BlackboxError(response['error'])

        return response
