# encoding: UTF-8

import urlparse

import requests
from ws_properties.utils.logs import get_logger_for_instance

from appcore.web.hooks import ResponseLoggerHook
from appcore.web.request_id import get_request_id


class DirectoryClient(object):
    def __init__(self, url, verify, ca, authenticator):
        self._logger = get_logger_for_instance(self)
        self._url = url
        self._session = requests.session()
        self._session.hooks['response'] = ResponseLoggerHook(self._logger)
        if verify:
            self._session.verify = ca or verify
        else:
            self._session.verify = False
        self._session.auth = authenticator

    def _make_url(self, url):
        return urlparse.urljoin(self._url, url.lstrip('/'))

    def list_domains(self, org_id, include_technical):
        request = self._session.get(
            self._make_url('/v7/domains/?fields=owned,tech'),
            headers={
                'X-Org-ID': str(org_id),
                'X-Request-ID': get_request_id(),
            },
        )
        with request as response:
            response.raise_for_status()
            domains = response.json()
        domains = [
            domain['name'] + '.'
            for domain in domains
            if domain['owned'] and (not domain['tech'] or include_technical)
        ]
        return list(
            map(lambda d: d.encode('idna'), domains)
        )

    def is_admin(self, org_id, uid):
        request = self._session.get(
            self._make_url('/permissions/'),
            headers={
                'X-Org-ID': str(org_id),
                'X-UID': str(uid),
                'X-User-IP': '127.0.0.1',
                'X-Request-ID': get_request_id(),
            },
        )
        with request as response:
            if response.status_code == 200:
                return 'add_domains' in response.json()
            elif response.status_code in (403, 404):
                return False
            else:
                response.raise_for_status()


class MockDirectoryClient(object):
    def list_domains(self, org_id, include_technical):
        domains = [
            u'example.com.',
            u'example.ru.',
            u'домен.рф.',
            u'новыйдомен1.рф.',
            u'новыйдомен2.рф.',
            u'example.yaconnect.com.',
        ]
        return list(
            map(lambda d: d.encode('idna'), domains)
        )

    def is_admin(self, org_id, uid):
        return bool(uid)
