# encoding: UTF-8

import urlparse
import requests
from ws_properties.utils.logs import get_logger_for_instance


class FourasError(Exception):
    pass


class FourasNotFoundError(FourasError):
    pass


# https://wiki.yandex-team.ru/mail/pdd/fouras/
class FourasClient(object):
    def __init__(self, url, verify, ca, authenticator):
        self._logger = get_logger_for_instance(self)
        self._url = url
        self._session = requests.session()
        if verify:
            self._session.verify = ca or verify
        else:
            self._session.verify = False
        self._session.auth = authenticator

    def _make_url(self, url):
        return urlparse.urljoin(self._url, url.lstrip('/'))

    def _raise_on_errors(self, response):
        if response.status_code == 404:
            raise FourasNotFoundError
        if response.status_code >= 400:
            self._logger.error('Fouras calling error (status_code=%s) (body=%s)', response.status_code, response.text)
            raise FourasError
        data = response.json()
        if data['status'] != 'ok':
            raise FourasError(data['response']['code'])
        return response

    def domain_key_gen(self, domain, selector=None):
        """
        {
           "response" : {
              "public_key" : "\"v=DKIM1; k=rsa; t=s; p=..."
              "private_key": "....",
              "domain": "yandex.ru",
              "selector": "mail",
              "enabled": true
           },
           "status" : "ok"
        }
        """
        data = {
            'domain': domain,
            'selector': selector or 'mail'
        }
        request = self._session.post(
            self._make_url('/domain/key/gen'),
            json=data
        )
        with request as response:
            response = self._raise_on_errors(response)
        return response.json()['response']

    def domain_status(self, domain, selector=None):
        """
        {
           "response" : {
              "enabled" : false,
              "domain" : "music.yandex.ru",
              "changed" : true,
              "selector" : "mail",
              "public_key" : "v=DKIM1; k=rsa; t=s; p=MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDdhGCVj1ITWT4P5OCKvN2vkHL98uiJCZ2WrCkB4m8lp1byKI55zd9PvgaxBIwJ8nQbJRax5kntktm7rXyQwv3wLjhzvZTQzDIZgrvAxvN2W8U0qt9hrdAnGLRRR5vS20aV1bl86pJMllyJJ1Wrk/foATor2jGV5sOAEs8KyQi+2QIDAQAB"
           },
           "status" : "ok"
        }
        """
        params = {
            'domain': domain,
            'selector': selector or 'mail'
        }
        request = self._session.get(
            self._make_url('/domain/status'),
            params=params
        )
        with request as response:
            response = self._raise_on_errors(response)
        return response.json()['response']


class MockFourasClient(object):
    def domain_status(self, domain, selector=None):
       return {
           'enabled': True,
           'changed': False,
           'public_key': 'v=DKIM1; k=rsa; t=s; p=MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDj1ppQM+bFhPHbTnSxGe73N277Ndz4MgRUzEEgzTzBRbfG6HTy8eBbbK6KoCP14fFp0D\/ZJMWZw71qoS6GJOG77qFH1677NXKIKqdu2Wripmyssjn0b5JZxdZSVcSGsw1JxnvTnsAMnxL+Rvq9BtwLiBLy\/zlvJYhjyPSHj2zFswIDAQAB',
           'domain': domain,
           'selector': selector or 'mail',
       }
