# encoding: UTF-8

from __future__ import absolute_import

import dns_hosting.services.auth.blackbox
import flask
import blackbox

from appcore.security.model import AuthIdentity
from appcore.security.model import HEADER_AUTHORIZATION
from appcore.security.model import RemoteIdentity
from appcore.security.provider import AuthProvider
from dns_hosting.services.auth.blackbox import JsonBlackbox
from dns_hosting.services.auth.providers.exceptions import ExpiredOAuthTokenError
from dns_hosting.services.auth.providers.exceptions import InvalidOAuthTokenError
from dns_hosting.services.auth.model import Scope


class OAuthProvider(AuthProvider):
    def __init__(self, blackbox_client):
        self.blackbox_client = blackbox_client  # type: JsonBlackbox

    def _get_oauth_token(self, request):
        auth_header = request.headers.get(HEADER_AUTHORIZATION)

        if auth_header is None:
            return

        parts = auth_header.strip().split(None, 2)
        parts += [None] * (2 - len(parts))
        auth_type, auth_data = parts

        if auth_type.lower() != 'oauth':
            return

        if not auth_data:
            raise InvalidOAuthTokenError()

        return auth_data

    def authenticate_request(
            self,
            request,  # type: flask.Request
            remote_identity,  # type: RemoteIdentity
    ):
        oauth_token = self._get_oauth_token(request)
        if oauth_token is None:
            return

        try:
            response = self.blackbox_client.oauth(
                oauth_token=oauth_token,
                userip=remote_identity.real_ip,
            )
        except blackbox.BlackboxError as e:
            if e.message == 'expired_token':
                raise ExpiredOAuthTokenError()
            else:
                raise

        return AuthIdentity(
            client_id=response['oauth']['client_id'],
            client_scopes=Scope.extract_any(
                response['oauth']['scope'].strip().split(),
            ),
            user_id=int(response['oauth']['uid']),
        )
