# encoding: UTF-8

from __future__ import absolute_import

import flask
import tvm2

from appcore.security.model import AuthIdentity
from appcore.security.model import RemoteIdentity
from appcore.security.provider import AuthProvider
from dns_hosting.services.auth.model import HEADER_YA_SERVICE_TICKET
from dns_hosting.services.auth.model import HEADER_YA_USER_TICKET
from dns_hosting.services.auth.model import Scope
from dns_hosting.services.auth.providers.exceptions import \
    InvalidServiceTicketError, InvalidUIDSpecifiedError
from dns_hosting.services.auth.providers.exceptions import InvalidUserTicketError
from dns_hosting.services.auth.providers.exceptions import NoUserIpError


class Tvm2AuthProvider(AuthProvider):
    def __init__(self, tvm2_client, client_scopes):
        self.tvm2_client = tvm2_client  # type: tvm2.TVM2
        self.client_scopes = client_scopes

    def _get_service_ticket(self, request):
        service_ticket_header = request.headers.get(HEADER_YA_SERVICE_TICKET)
        if service_ticket_header is None:
            return

        service_ticket = self.tvm2_client.parse_service_ticket(
            service_ticket_header.strip(),
        )
        if service_ticket is None:
            raise InvalidServiceTicketError()

        return service_ticket

    def _get_user_ticket(self, request):
        user_ticket_header = request.headers.get(HEADER_YA_USER_TICKET)
        if user_ticket_header is None:
            return

        user_ticket = self.tvm2_client.parse_user_ticket(
            user_ticket_header.strip(),
        )
        if user_ticket is None:
            raise InvalidUserTicketError()

        return user_ticket

    def _get_x_uid_value(self, request):
        x_uid = request.headers.get('X-UID')

        if x_uid is not None:
            try:
                x_uid = int(x_uid)
            except ValueError:
                raise InvalidUIDSpecifiedError()

        return x_uid

    def authenticate_request(
            self,
            request,  # type: flask.Request
            remote_identity,  # type: RemoteIdentity
    ):
        service_ticket = self._get_service_ticket(request)
        if service_ticket is None:
            return

        client_id = service_ticket.src
        # client_scopes = Scope.extract_any(service_ticket.scopes)
        client_scopes = set(self.client_scopes.get(client_id, []))

        x_uid = self._get_x_uid_value(request)
        user_ticket = self._get_user_ticket(request)
        user_id = None

        if user_ticket is None:
            if Scope.authorize_user_by_xuid in client_scopes:
                user_id = x_uid
        else:
            if x_uid is None or x_uid not in user_ticket.uids:
                user_id = user_ticket.default_uid
            else:
                user_id = x_uid

        if user_id is not None and not remote_identity.user_ip:
            raise NoUserIpError()

        return AuthIdentity(
            client_id=client_id,
            client_scopes=client_scopes,
            user_id=user_id,
        )
