# encoding: UTF-8

import os
import signal

import gevent
import gevent.pool
from ws_properties.utils.logs import get_logger_for_instance

from dns_hosting.services.dnsserver.nsd.ctl import NSDControl
from dns_hosting.services.dnsserver.nsd.supervisor import NSDSupervisor
from dns_hosting.services.dnsserver.properties import NsdServerProperties
from dns_hosting.services.dnsserver.zonetracker import ZonesTracker
from dns_hosting.services.support.retry import StandardRetryPolicy, retry
from dns_hosting.utils.iterators import ijoin
from dns_hosting.utils.progress import ProgressLogger


class NsdDnsServer(object):
    def __init__(self, properties):
        # type: (NsdServerProperties) -> None

        super(NsdDnsServer, self).__init__()

        self._properties = properties  # type:
        self._supervisor = NSDSupervisor(
            config_file=properties.config_file,
            executable=properties.executable,
            graceful_shutdown_timeout=properties.graceful_shutdown_timeout,
        )
        self._control = NSDControl(
            address=(properties.control.host, properties.control.port),
            keyfile=properties.control.keyfile,
            certfile=properties.control.certfile,
        )
        self._zones_tracker = ZonesTracker(
            host=properties.dnsmaster.http.host,
            port=properties.dnsmaster.http.port,
            sync_interval=properties.dnsmaster.http.sync_interval,
        )
        self._zone_names = []
        self._current_task = None  # type: gevent.Greenlet
        self._logger = get_logger_for_instance(self)

    def serve_forever(self):
        try:
            gevent.signal(signal.SIGINT, self._supervisor.stop)
            gevent.signal(signal.SIGTERM, self._supervisor.stop)

            self._supervisor.start(self._on_start, self._on_exit)

            gevent.wait()
        finally:
            self._logger.info('Service terminated.')

    def _update_zone_names(self):
        zone_names = []

        if os.path.exists(self._properties.zone_file):
            with open(self._properties.zone_file) as f:
                for line in f:
                    if line.startswith('#'):
                        continue

                    action, zone_name, pattern = line.split(' ', 3)
                    if action == 'add':
                        zone_names.append(zone_name)

        zone_names.sort()
        self._zone_names = zone_names

    def _on_start(self, supervisor, error):
        self._update_zone_names()
        self._zones_tracker.start(self._on_zone_names_change)

    def _on_exit(self, supervisor, exitcode):
        self._zones_tracker.stop()
        if self._current_task:
            self._current_task.kill(block=False)

    def _on_zone_names_change(self, zone_tracker, error, zone_names):
        if not error and not self._current_task:
            self._current_task = gevent.spawn(
                self._actualize_zones,
                zone_names,
            )

    def _actualize_zones(self, zone_names):
        local_names_it = iter(self._zone_names)
        remote_names_it = iter(zone_names)

        added = []
        removed = []
        for local_name, remote_name in ijoin(local_names_it, remote_names_it):
            if local_name is None and remote_name is None:
                raise AssertionError('Illegal state error.')
            elif remote_name is None:
                removed.append(local_name)
            elif local_name is None:
                added.append(remote_name)
            else:
                pass  # nothing to do

        if added or removed:
            self._apply_changes(added, removed)

    def _apply_changes(self, added, removed):
        self._logger.info(
            'Found changes (new=%d, obsolete=%d).',
            len(added),
            len(removed),
        )

        tmp_zone_names = set(self._zone_names)
        try:
            progress = ProgressLogger(
                message='Applying zones changes',
                logger=self._logger,
            )
            with progress:
                retry_policy = StandardRetryPolicy(delay=1, max_attempts=60)
                chunk_size = 100000

                for offset in xrange(0, len(removed), chunk_size):
                    for attempt in retry(retry_policy):
                        with attempt:
                            chunk = removed[offset:offset + chunk_size]
                            self._control.delzones(chunk)
                            progress.update(len(chunk))
                            tmp_zone_names -= set(removed)

                for offset in xrange(0, len(added), chunk_size):
                    for attempt in retry(retry_policy):
                        with attempt:
                            chunk = added[offset:offset + chunk_size]
                            self._control.addzones(chunk)
                            progress.update(len(chunk))
                            tmp_zone_names |= set(added)

            self._zone_names = sorted(tmp_zone_names)
        except Exception:
            self._logger.exception('Zone changes failed.')
        else:
            self._logger.info('Zone changes successfully applied.')
